package account

import (
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	serviceMembers "git.perx.ru/perxis/perxis-go/pkg/members/middleware"
	membersObserverTransport "git.perx.ru/perxis/perxis-go/pkg/members/observer/transport/grpc"
	membersTransport "git.perx.ru/perxis/perxis-go/pkg/members/transport/grpc"
	serviceOrganizations "git.perx.ru/perxis/perxis-go/pkg/organizations/middleware"
	organizationsTransport "git.perx.ru/perxis/perxis-go/pkg/organizations/transport/grpc"
	serviceUsers "git.perx.ru/perxis/perxis-go/pkg/users/middleware"
	usersTransport "git.perx.ru/perxis/perxis-go/pkg/users/transport/grpc"
	"go.uber.org/zap"
	"google.golang.org/grpc"
)

const (
	DefaultCacheSize = 1000
	DefaultCacheTTL  = time.Second * 10
)

func NewClient(conn *grpc.ClientConn, opts ...Option) *Account {

	client := &Account{}
	c := &config{}

	for _, o := range opts {
		o(c)

	}

	if c.logger == nil {
		c.logger = zap.NewNop()
	}

	client.Members = membersTransport.NewGRPCClient(conn, "", c.clientOptions...)
	client.Organizations = organizationsTransport.NewGRPCClient(conn, "", c.clientOptions...)
	client.Users = usersTransport.NewGRPCClient(conn, "", c.clientOptions...)
	client.MembersObserver = membersObserverTransport.NewGRPCClient(conn, "", c.clientOptions...)

	if !c.noCache {
		client = WithCaching(client, DefaultCacheSize, DefaultCacheTTL)
	}

	if !c.noLog {
		client = WithLogging(client, c.logger, c.accessLog)
	}

	return client
}

func WithCaching(client *Account, size int, ttl time.Duration) *Account {
	c := *client

	c.Members = serviceMembers.CachingMiddleware(cache.NewCache(size, ttl))(client.Members)
	c.Organizations = serviceOrganizations.CachingMiddleware(cache.NewCache(size, ttl))(client.Organizations)
	c.Users = serviceUsers.CachingMiddleware(cache.NewCache(size, ttl))(client.Users)

	return &c
}

func WithLogging(client *Account, logger *zap.Logger, accessLog bool) *Account {
	c := *client

	c.Members = serviceMembers.WithLog(c.Members, logger, accessLog)
	c.Organizations = serviceOrganizations.WithLog(c.Organizations, logger, accessLog)
	c.Users = serviceUsers.WithLog(c.Users, logger, accessLog)

	return &c
}
