package template

import (
	"context"
	"errors"
	"testing"
	"text/template"

	"git.perx.ru/perxis/perxis-go/pkg/content"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	mocksitems "git.perx.ru/perxis/perxis-go/pkg/items/mocks"
	"github.com/stretchr/testify/assert"
)

func TestBuilder_Execute(t *testing.T) {
	tests := []struct {
		name    string
		ctx     context.Context
		cnt     *content.Content
		SpaceID string
		EnvID   string
		funcs   template.FuncMap
		str     string
		data    any
		want    any
		wantErr bool

		itemsCall func(itemsSvc *mocksitems.Items)
	}{
		{name: "error", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ .a }}", data: "world", want: "", wantErr: true},
		{name: "empty", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "", data: "", want: "", wantErr: false},
		{name: "#1", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ . }}", data: "world", want: "hello world", wantErr: false},
		{name: "#2", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "{{ . }}", data: "world", want: "world", wantErr: false},
		{name: "#3 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "", data: "world", want: "", wantErr: false},
		{name: "#4 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello", data: "world", want: "hello", wantErr: false},
		{name: "lookup", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello, {{ lookup \"secrets.dev.key\" }}", data: "", want: "hello, Luk", wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "Luk",
				},
			}, nil).Once()
		}},
		{name: "lookup with slice", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "numbers {{ lookup \"secrets.dev.slice\" }}", data: "", want: "numbers [1 2 3]", wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":    "dev",
					"slice": []int{1, 2, 3},
				},
			}, nil).Once()
		}},
		{name: "lookup with empty Data", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "numbers {{ lookup \"secrets.dev.slice\" }}", data: "", want: "numbers <no value>", wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data:         map[string]interface{}{},
			}, nil).Once()
		}},
		{name: "lookup with incorrect field", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ lookup \"secrets.dev.incorrect\" }}", data: "", want: "hello <no value>", wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "1234",
				},
			}, nil).Once()
		}},
		{name: "lookup not found", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ lookup \"secrets.prod.pass\" }}", data: "", want: "", wantErr: true, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "prod").Return(nil, errors.New("not found")).Once()
		}},
		{name: "lookup without itemID", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ lookup \"secrets.pass\" }}", data: "", want: "", wantErr: true},
		{name: "system ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: "hello {{ system.SpaceID }}", data: "", want: "hello space", wantErr: false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			itemsSvc := &mocksitems.Items{}
			if tt.itemsCall != nil {
				tt.itemsCall(itemsSvc)
			}
			tt.cnt = &content.Content{
				Items: itemsSvc,
			}
			b := &Builder{
				ctx:     tt.ctx,
				cnt:     tt.cnt,
				SpaceID: tt.SpaceID,
				EnvID:   tt.EnvID,
				funcs:   tt.funcs,
			}

			got, err := b.Execute(tt.str, tt.data)
			if tt.wantErr == true {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}
			assert.Equal(t, tt.want, got)
			if tt.itemsCall != nil {
				itemsSvc.AssertExpectations(t)
			}
		})
	}
}

func TestBuilder_ExecuteList(t *testing.T) {
	tests := []struct {
		name    string
		ctx     context.Context
		cnt     *content.Content
		SpaceID string
		EnvID   string
		funcs   template.FuncMap
		str     []string
		data    any
		want    []string
		wantErr bool

		itemsCall func(itemsSvc *mocksitems.Items)
	}{
		{name: "error", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello { . }}", "go {{ . }"}, data: "world", want: []string{}, wantErr: true},
		{name: "empty", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{""}, data: "world", want: []string{""}, wantErr: false},
		{name: "#1", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello {{ . }}", "go {{ . }}"}, data: "world", want: []string{"hello world", "go world"}, wantErr: false},
		{name: "#2", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"{{ . }}"}, data: "world", want: []string{"world"}, wantErr: false},
		{name: "#3 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{""}, data: "world", want: []string{""}, wantErr: false},
		{name: "#4 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello"}, data: "world", want: []string{"hello"}, wantErr: false},
		{name: "lookup", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello {{ lookup \"secrets.dev.key\" }}"}, data: "", want: []string{"hello 1234"}, wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "1234",
				},
			}, nil).Once()
		}},
		{name: "lookup with incorrect field", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello {{ lookup \"secrets.dev.incorrect\" }}"}, data: "", want: []string{"hello <no value>"}, wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "1234",
				},
			}, nil).Once()
		}},
		{name: "system ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: []string{"hello {{ system.SpaceID }}"}, data: "", want: []string{"hello space"}, wantErr: false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			itemsSvc := &mocksitems.Items{}
			if tt.itemsCall != nil {
				tt.itemsCall(itemsSvc)
			}
			tt.cnt = &content.Content{
				Items: itemsSvc,
			}
			b := &Builder{
				ctx:     tt.ctx,
				cnt:     tt.cnt,
				SpaceID: tt.SpaceID,
				EnvID:   tt.EnvID,
				funcs:   tt.funcs,
			}

			got, err := b.ExecuteList(tt.str, tt.data)
			if tt.wantErr == true {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}
			assert.Equal(t, tt.want, got)
			if tt.itemsCall != nil {
				itemsSvc.AssertExpectations(t)
			}
		})
	}
}

func TestBuilder_ExecuteMap(t *testing.T) {
	tests := []struct {
		name    string
		ctx     context.Context
		cnt     *content.Content
		SpaceID string
		EnvID   string
		funcs   template.FuncMap
		str     map[string]interface{}
		data    any
		want    map[string]interface{}
		wantErr bool

		itemsCall func(itemsSvc *mocksitems.Items)
	}{
		{name: "error", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ . }"}, data: "world", want: nil, wantErr: true},
		{name: "empty", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{}, data: "", want: map[string]interface{}{}, wantErr: false},
		{name: "#1", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ . }}"}, data: "world", want: map[string]interface{}{"hello": "world"}, wantErr: false},
		{name: "#2", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ . }}", "go": "{{ . }}"}, data: "world", want: map[string]interface{}{"hello": "world", "go": "world"}, wantErr: false},
		{name: "#3 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{}, data: "world", want: map[string]interface{}{}, wantErr: false},
		{name: "#4 ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"a": "b"}, data: "world", want: map[string]interface{}{"a": "b"}, wantErr: false},
		{name: "lookup ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ lookup \"secrets.dev.key\" }}"}, data: "", want: map[string]interface{}{"hello": "1234"}, wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "1234",
				},
			}, nil).Once()
		}},
		{name: "lookup with incorrect field", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ lookup \"secrets.dev.incorrect\" }}"}, data: "", want: map[string]interface{}{"hello": "<no value>"}, wantErr: false, itemsCall: func(itemsSvc *mocksitems.Items) {
			itemsSvc.On("Get", context.Background(), "space", "env", "secrets", "dev").Return(&items.Item{
				ID:           "dev",
				SpaceID:      "space",
				EnvID:        "env",
				CollectionID: "secrets",
				Data: map[string]interface{}{
					"id":  "dev",
					"key": "1234",
				},
			}, nil).Once()
		}},
		{name: "system ", ctx: context.Background(), cnt: &content.Content{}, SpaceID: "space", EnvID: "env", funcs: template.FuncMap{}, str: map[string]interface{}{"hello": "{{ system.SpaceID }}"}, data: "", want: map[string]interface{}{"hello": "space"}, wantErr: false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			itemsSvc := &mocksitems.Items{}
			if tt.itemsCall != nil {
				tt.itemsCall(itemsSvc)
			}
			tt.cnt = &content.Content{
				Items: itemsSvc,
			}
			b := &Builder{
				ctx:     tt.ctx,
				cnt:     tt.cnt,
				SpaceID: tt.SpaceID,
				EnvID:   tt.EnvID,
				funcs:   tt.funcs,
			}

			got, err := b.ExecuteMap(tt.str, tt.data)
			if tt.wantErr == true {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}
			assert.Equal(t, tt.want, got)
			if tt.itemsCall != nil {
				itemsSvc.AssertExpectations(t)
			}
		})
	}
}
