package middleware

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/spaces"
)

func orgKey(orgID string) string { return "org-" + orgID }

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Spaces) service.Spaces {
		m := &cachingMiddleware{
			cache: cache,
			next:  next,
		}

		return m
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Spaces
}

func (m cachingMiddleware) Create(ctx context.Context, space *service.Space) (sp *service.Space, err error) {

	sp, err = m.next.Create(ctx, space)
	if err == nil {
		m.cache.Remove(orgKey(sp.OrgID))
	}
	return sp, err
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId string) (sp *service.Space, err error) {

	value, e := m.cache.Get(spaceId)
	if e == nil {
		return value.(*service.Space), err
	}
	sp, err = m.next.Get(ctx, spaceId)
	if err == nil {
		m.cache.Set(spaceId, sp)
	}
	return sp, err
}

func (m cachingMiddleware) List(ctx context.Context, orgId string) (spaces []*service.Space, err error) {

	value, e := m.cache.Get(orgKey(orgId))
	if e == nil {
		return value.([]*service.Space), err
	}
	spaces, err = m.next.List(ctx, orgId)
	if err == nil {
		m.cache.Set(orgKey(orgId), spaces)
		for _, s := range spaces {
			m.cache.Set(s.ID, s)
		}
	}
	return spaces, err
}

func (m cachingMiddleware) Update(ctx context.Context, space *service.Space) (err error) {

	err = m.next.Update(ctx, space)
	if err == nil {
		value, e := m.cache.Get(space.ID)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
		}
		m.cache.Remove(space.ID)
	}
	return err
}

func (m cachingMiddleware) UpdateConfig(ctx context.Context, spaceId string, config *service.Config) (err error) {

	err = m.next.UpdateConfig(ctx, spaceId, config)
	if err == nil {
		value, e := m.cache.Get(spaceId)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func (m cachingMiddleware) Delete(ctx context.Context, spaceId string) (err error) {

	err = m.next.Delete(ctx, spaceId)
	if err == nil {
		value, e := m.cache.Get(spaceId)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func (m cachingMiddleware) Transfer(ctx context.Context, spaceID, transferToOrg string) error {
	err := m.next.Transfer(ctx, spaceID, transferToOrg)
	if err == nil {
		value, e := m.cache.Get(spaceID)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
			m.cache.Remove(orgKey(space.TransferToOrg))
		}
		m.cache.Remove(spaceID)
		m.cache.Remove(transferToOrg)
	}
	return err
}

func (m cachingMiddleware) AbortTransfer(ctx context.Context, spaceID string) error {
	err := m.next.AbortTransfer(ctx, spaceID)
	if err == nil {
		value, e := m.cache.Get(spaceID)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
			m.cache.Remove(orgKey(space.TransferToOrg))
		}
		m.cache.Remove(spaceID)
	}
	return err
}

func (m cachingMiddleware) ListTransfers(ctx context.Context, orgID string) (spaces []*service.Space, err error) {
	return m.next.ListTransfers(ctx, orgID)
}

func (m cachingMiddleware) Move(ctx context.Context, spaceID, orgID string) error {
	err := m.next.Move(ctx, spaceID, orgID)
	if err == nil {
		value, e := m.cache.Get(spaceID)
		if e == nil {
			space := value.(*service.Space)
			m.cache.Remove(orgKey(space.OrgID))
			m.cache.Remove(orgKey(space.TransferToOrg))
		}
		m.cache.Remove(spaceID)
		m.cache.Remove(orgID)
	}
	return err
}
