package setup

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/clients"
	clientsMock "git.perx.ru/perxis/perxis-go/pkg/clients/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/collections"
	collectionMock "git.perx.ru/perxis/perxis-go/pkg/collections/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/content"
	"git.perx.ru/perxis/perxis-go/pkg/data"
	environmentMock "git.perx.ru/perxis/perxis-go/pkg/environments/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/extension"
	"git.perx.ru/perxis/perxis-go/pkg/items"
	itemsMock "git.perx.ru/perxis/perxis-go/pkg/items/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
	rolesMock "git.perx.ru/perxis/perxis-go/pkg/roles/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"go.uber.org/zap/zaptest"
)

const (
	spaceID = "sp"
	envID   = "env"
)

func getCollections() []*collections.Collection {
	return []*collections.Collection{
		{
			ID:      "coll1",
			SpaceID: spaceID,
			EnvID:   envID,
			Schema:  schema.New(),
			Name:    "Коллекция",
		},
	}
}

func getRoles() []*roles.Role {
	return []*roles.Role{
		{
			ID:      "role",
			SpaceID: spaceID,
		},
	}
}

func getClients() []*clients.Client {
	return []*clients.Client{
		{
			ID:      "client",
			SpaceID: spaceID,
			RoleID:  "role",
		},
	}
}

func getActions() []*items.Item {
	return []*items.Item{
		{
			ID:           "act",
			SpaceID:      spaceID,
			EnvID:        envID,
			CollectionID: extension.ActionsCollectionID,
			Data: map[string]interface{}{
				"action":    "act",
				"name":      "Action",
				"extension": "ext",
			},
		},
	}
}

func newSetup(content *content.Content, t *testing.T) *Setup {
	logger := zaptest.NewLogger(t, zaptest.WrapOptions())

	setup := NewSetup(content, spaceID, envID, logger)
	setup.AddCollections(getCollections())
	setup.AddRoles(getRoles())
	setup.AddClients(getClients())
	setup.AddItems(getActions(), OverwriteItem())

	return setup
}

func TestSetupInstall(t *testing.T) {
	t.Run("Success, nothing to install", func(t *testing.T) {
		logger := zaptest.NewLogger(t, zaptest.WrapOptions())

		setup := NewSetup(nil, spaceID, envID, logger)

		err := setup.Install(context.Background())

		require.NoError(t, err)
	})

	t.Run("Success, no force", func(t *testing.T) {
		envMocks := &environmentMock.Environments{}
		envMocks.On("Migrate", mock.Anything, spaceID, envID).
			Return(nil).
			Once()

		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, collections.ErrNotFound).
				Once()

			collsMock.On("Create", mock.Anything, collection).
				Return(collection, nil).
				Once()

			collsMock.On("SetSchema", mock.Anything, spaceID, envID, collection.ID, collection.Schema).
				Return(nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).Run(func(args mock.Arguments) {
				create := args[1].(*roles.Role)
				require.True(t, data.Contains(envID, create.Environments))
			}).Return(role, nil).Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itmMock.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
			Return(nil, 0, nil).
			Once()
		for _, act := range getActions() {
			itmMock.On("Create", mock.Anything, act).
				Return(act, nil).
				Once()
			itmMock.On("Publish", mock.Anything, act).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections:  collsMock,
			Clients:      clMock,
			Roles:        rMock,
			Items:        itmMock,
			Environments: envMocks,
		}, t)

		err := setup.Install(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
		envMocks.AssertExpectations(t)
	})

	t.Run("Success, update existing records", func(t *testing.T) {
		envMocks := &environmentMock.Environments{}
		collsMock := &collectionMock.Collections{}
		collsMock.On("Get", mock.Anything, spaceID, envID, "coll1").
			Return(&collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция старая"}, nil).Once()
		collsMock.On("Update", mock.Anything, &collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция"}).Return(nil).Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(role, nil).
				Once()

			rMock.On("Update", mock.Anything, role).
				Return(nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(client, nil).
				Once()

			clMock.On("Update", mock.Anything, client).
				Return(nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itms := getActions()
		itmMock.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
			Return(itms, 0, nil).
			Once()
		itmMock.On("Update", mock.Anything, mock.Anything).
			Return(nil).
			Times(len(itms))
		itmMock.On("Publish", mock.Anything, mock.Anything).
			Return(nil).
			Times(len(itms))

		setup := newSetup(&content.Content{
			Collections:  collsMock,
			Clients:      clMock,
			Roles:        rMock,
			Items:        itmMock,
			Environments: envMocks,
		}, t)

		err := setup.Install(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
		envMocks.AssertExpectations(t)
	})

	t.Run("Success, with force", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		collsMock.On("Get", mock.Anything, spaceID, envID, "coll1").
			Return(&collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция старая"}, nil).Once()
		collsMock.On("Update", mock.Anything, &collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция"}).Return(nil).Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).Return(role, nil).Once()
			rMock.On("Update", mock.Anything, &roles.Role{ID: "role", SpaceID: "sp", Environments: []string{"env"}}).Return(nil).Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itmMock.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
			Return(nil, 0, nil).
			Once()
		for _, act := range getActions() {
			itmMock.On("Create", mock.Anything, act).
				Return(act, nil).
				Once()
			itmMock.On("Publish", mock.Anything, act).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)
		setup = setup.WithForce(true)

		err := setup.Install(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
	})

	t.Run("Can't install role, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}
		rMock.On("Get", mock.Anything, spaceID, mock.Anything).
			Return(nil, errors.New("can't get role")).
			Once()

		setup := newSetup(&content.Content{
			Roles: rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install role")

		rMock.AssertExpectations(t)
	})

	t.Run("Can't install client, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, errors.New("can't get client")).
				Once()
		}

		setup := newSetup(&content.Content{
			Clients: clMock,
			Roles:   rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install client")

		rMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't get collection, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, errors.New("can't get collection")).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install collection")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't create collection, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, collections.ErrNotFound).
				Once()

			collsMock.On("Create", mock.Anything, collection).
				Return(nil, errors.New("can't create collection")).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install collection")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't update collection, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		collsMock.On("Get", mock.Anything, spaceID, envID, "coll1").Return(&collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция 00"}, nil).Once()
		collsMock.On("Update", mock.Anything, &collections.Collection{ID: "coll1", SpaceID: spaceID, EnvID: envID, Schema: schema.New(), Name: "Коллекция"}).Return(errors.New("can't update collection")).Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install collection")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't set schema, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, collections.ErrNotFound).
				Once()

			collsMock.On("Create", mock.Anything, collection).
				Return(collection, nil).
				Once()

			collsMock.On("SetSchema", mock.Anything, spaceID, envID, collection.ID, collection.Schema).
				Return(errors.New("can't set schema")).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install collection")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't migrate, storage returns error", func(t *testing.T) {
		envMocks := &environmentMock.Environments{}
		envMocks.On("Migrate", mock.Anything, spaceID, envID).
			Return(errors.New("can't migrate")).
			Once()

		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, collections.ErrNotFound).
				Once()

			collsMock.On("Create", mock.Anything, collection).
				Return(collection, nil).
				Once()

			collsMock.On("SetSchema", mock.Anything, spaceID, envID, collection.ID, collection.Schema).
				Return(nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(nil, roles.ErrNotFound).
				Once()

			rMock.On("Create", mock.Anything, mock.Anything).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(nil, clients.ErrNotFound).
				Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections:  collsMock,
			Clients:      clMock,
			Roles:        rMock,
			Environments: envMocks,
		}, t)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "migrate: can't migrate")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't find action, storage returns error", func(t *testing.T) {
		envMocks := &environmentMock.Environments{}
		envMocks.On("Migrate", mock.Anything, spaceID, envID).
			Return(nil).
			Once()

		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(nil, collections.ErrNotFound).
				Once()

			collsMock.On("Create", mock.Anything, collection).
				Return(collection, nil).
				Once()

			collsMock.On("SetSchema", mock.Anything, spaceID, envID, collection.ID, collection.Schema).
				Return(nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).Return(role, nil).Once()
			rMock.On("Update", mock.Anything, &roles.Role{ID: "role", SpaceID: "sp", Environments: []string{"env"}}).Return(nil).Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()

			clMock.On("Create", mock.Anything, client).
				Return(client, nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itmMock.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
			Return(nil, 0, nil).
			Once()

		itmMock.On("Create", mock.Anything, mock.Anything).
			Return(nil, errors.New("can't create item")).
			Once()

		setup := newSetup(&content.Content{
			Collections:  collsMock,
			Clients:      clMock,
			Roles:        rMock,
			Items:        itmMock,
			Environments: envMocks,
		}, t)
		setup = setup.WithForce(true)

		err := setup.Install(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to install item")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
		envMocks.AssertExpectations(t)
	})

	//t.Run("Can't find task configs, storage returns error", func(t *testing.T) {
	//	envMocks := &environmentMock.Environments{}
	//	envMocks.On("Migrate", mock.Anything, spaceID, envID).
	//		Return(nil).
	//		Once()
	//
	//	collsMock := &collectionMock.Collections{}
	//	for _, collection := range getCollections() {
	//		collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
	//			Return(nil, collections.ErrNotFound).
	//			Once()
	//
	//		collsMock.On("Create", mock.Anything, collection).
	//			Return(collection, nil).
	//			Once()
	//
	//		collsMock.On("SetSchema", mock.Anything, spaceID, envID, collection.ID, collection.Schema).
	//			Return(nil).
	//			Once()
	//	}
	//
	//	rMock := &rolesMock.Roles{}
	//	for _, role := range getRoles() {
	//		rMock.On("Get", mock.Anything, spaceID, role.ID).
	//			Return(nil, roles.ErrNotFound).
	//			Once()
	//
	//		rMock.On("Create", mock.Anything, mock.Anything).
	//			Return(role, nil).
	//			Once()
	//	}
	//
	//	clMock := &clientsMock.Clients{}
	//	for _, client := range getClients() {
	//		clMock.On("Get", mock.Anything, spaceID, client.ID).
	//			Return(nil, clients.ErrNotFound).
	//			Once()
	//
	//		clMock.On("Create", mock.Anything, client).
	//			Return(client, nil).
	//			Once()
	//	}
	//
	//	itmMock := &itemsMock.Items{}
	//	for _, act := range getActions() {
	//		itmMock.On("Get", mock.Anything, spaceID, envID, extension.ActionsCollectionID, act.ID).
	//			Return(nil, items.ErrNotFound).
	//			Once()
	//
	//		itmMock.On("Create", mock.Anything, act).
	//			Return(act, nil).
	//			Once()
	//		itmMock.On("Publish", mock.Anything, act).
	//			Return(nil).
	//			Once()
	//	}
	//
	//	setup := newSetup(&content.Content{
	//		Collections:  collsMock,
	//		Clients:      clMock,
	//		Roles:        rMock,
	//		Items:        itmMock,
	//		Environments: envMocks,
	//	}, t)
	//
	//	err := setup.Install(context.Background())
	//
	//	require.Error(t, err)
	//	assert.ErrorContains(t, err, "failed to install task configs")
	//
	//	rMock.AssertExpectations(t)
	//	collsMock.AssertExpectations(t)
	//	clMock.AssertExpectations(t)
	//	itmMock.AssertExpectations(t)
	//	envMocks.AssertExpectations(t)
	//})
}

func TestSetupUninstall(t *testing.T) {
	t.Run("Success, nothing to uninstall", func(t *testing.T) {
		logger := zaptest.NewLogger(t, zaptest.WrapOptions())

		setup := NewSetup(nil, spaceID, envID, logger)

		err := setup.Uninstall(context.Background())

		require.NoError(t, err)
	})

	t.Run("Remove", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Delete", mock.Anything, spaceID, envID, collection.ID).
				Return(nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Delete", mock.Anything, spaceID, role.ID).
				Return(nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()
		}

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, mock.Anything).Run(func(args mock.Arguments) {
				del := args[1].(*items.Item)
				require.Equal(t, spaceID, del.SpaceID)
				require.Equal(t, envID, del.EnvID)
				require.Equal(t, extension.ActionsCollectionID, del.CollectionID)
				require.Equal(t, act.ID, del.ID)
			}).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
	})

	t.Run("Remove, with clients NotFound error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Delete", mock.Anything, spaceID, envID, collection.ID).
				Return(nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Delete", mock.Anything, spaceID, role.ID).
				Return(nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		clMock.On("Delete", mock.Anything, spaceID, mock.Anything).
			Return(clients.ErrNotFound).
			Once()

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, mock.Anything).Run(func(args mock.Arguments) {
				del := args[1].(*items.Item)
				require.Equal(t, spaceID, del.SpaceID)
				require.Equal(t, envID, del.EnvID)
				require.Equal(t, extension.ActionsCollectionID, del.CollectionID)
				require.Equal(t, act.ID, del.ID)
			}).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
	})

	t.Run("Can't uninstall clients, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}

		clMock := &clientsMock.Clients{}
		clMock.On("Delete", mock.Anything, spaceID, mock.Anything).
			Return(errors.New("can't delete client")).
			Once()

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, spaceID, envID, extension.ActionsCollectionID, act.ID).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Clients: clMock,
			Roles:   rMock,
			Items:   itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to uninstall client")

		rMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't uninstall role, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}
		rMock.On("Delete", mock.Anything, spaceID, mock.Anything).
			Return(errors.New("can't delete role")).
			Once()

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()
		}

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, spaceID, envID, extension.ActionsCollectionID, act.ID).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Roles:   rMock,
			Clients: clMock,
			Items:   itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to uninstall role")

		rMock.AssertExpectations(t)
	})

	t.Run("Can't uninstall collections, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		collsMock.On("Delete", mock.Anything, spaceID, envID, mock.Anything).
			Return(errors.New("can't delete collection")).
			Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Delete", mock.Anything, spaceID, role.ID).
				Return(nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()
		}

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, spaceID, envID, extension.ActionsCollectionID, act.ID).
				Return(nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to uninstall collection")

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't uninstall actions, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		collsMock.On("Delete", mock.Anything, spaceID, envID, mock.Anything).
			Return(nil).
			Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Delete", mock.Anything, spaceID, role.ID).
				Return(nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Delete", mock.Anything, spaceID, client.ID).Return(nil).Once()
		}

		itmMock := &itemsMock.Items{}
		for _, act := range getActions() {
			itmMock.On("Delete", mock.Anything, mock.Anything).Run(func(args mock.Arguments) {
				del := args[1].(*items.Item)
				require.Equal(t, spaceID, del.SpaceID)
				require.Equal(t, envID, del.EnvID)
				require.Equal(t, extension.ActionsCollectionID, del.CollectionID)
				require.Equal(t, act.ID, del.ID)
			}).
				Return(errors.New("can't delete item")).
				Once()
		}

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)

		setup = setup.WithRemove(true)
		err := setup.Uninstall(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "failed to uninstall item")

		itmMock.AssertExpectations(t)
	})
}

func TestSetupCheck(t *testing.T) {
	t.Run("Success, nothing to check", func(t *testing.T) {
		logger := zaptest.NewLogger(t, zaptest.WrapOptions())

		setup := NewSetup(nil, spaceID, envID, logger)

		err := setup.Check(context.Background())

		require.NoError(t, err)
	})

	t.Run("Success", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(collection, nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(client, nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itmMock.On(
			"Find",
			mock.Anything,
			spaceID,
			envID,
			extension.ActionsCollectionID,
			mock.MatchedBy(func(filter *items.Filter) bool { return data.Contains("act", filter.ID) }),
			mock.MatchedBy(func(opt *items.FindOptions) bool { return opt.Regular && opt.Hidden && opt.Templates }),
		).Return(getActions(), 0, nil).Once()

		setup := newSetup(&content.Content{
			Collections: collsMock,
			Clients:     clMock,
			Roles:       rMock,
			Items:       itmMock,
		}, t)

		err := setup.Check(context.Background())

		require.NoError(t, err)
		require.False(t, setup.HasErrors())

		rMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		itmMock.AssertExpectations(t)
	})

	t.Run("Can't get role, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}
		rMock.On("Get", mock.Anything, spaceID, mock.Anything).
			Return(nil, errors.New("can't get role")).
			Once()

		setup := newSetup(&content.Content{
			Roles: rMock,
		}, t)

		err := setup.Check(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "role check error")

		rMock.AssertExpectations(t)
	})

	t.Run("Can't get client, storage returns error", func(t *testing.T) {
		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		clMock.On("Get", mock.Anything, spaceID, mock.Anything).
			Return(nil, errors.New("can't get client")).
			Once()

		setup := newSetup(&content.Content{
			Roles:   rMock,
			Clients: clMock,
		}, t)

		err := setup.Check(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "clients check error")

		rMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
	})

	t.Run("Can't get collection, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		collsMock.On("Get", mock.Anything, spaceID, envID, mock.Anything).
			Return(nil, errors.New("can't get collection")).
			Once()

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(client, nil).
				Once()
		}

		setup := newSetup(&content.Content{
			Roles:       rMock,
			Clients:     clMock,
			Collections: collsMock,
		}, t)

		err := setup.Check(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "collections check error")

		rMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
	})

	t.Run("Can't get action, storage returns error", func(t *testing.T) {
		collsMock := &collectionMock.Collections{}
		for _, collection := range getCollections() {
			collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
				Return(collection, nil).
				Once()
		}

		rMock := &rolesMock.Roles{}
		for _, role := range getRoles() {
			rMock.On("Get", mock.Anything, spaceID, role.ID).
				Return(role, nil).
				Once()
		}

		clMock := &clientsMock.Clients{}
		for _, client := range getClients() {
			clMock.On("Get", mock.Anything, spaceID, client.ID).
				Return(client, nil).
				Once()
		}

		itmMock := &itemsMock.Items{}
		itmMock.On("Find", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
			Return(nil, 0, nil).
			Once()

		setup := newSetup(&content.Content{
			Roles:       rMock,
			Clients:     clMock,
			Collections: collsMock,
			Items:       itmMock,
		}, t)

		err := setup.Check(context.Background())

		require.Error(t, err)
		assert.ErrorContains(t, err, "items check error")

		rMock.AssertExpectations(t)
		clMock.AssertExpectations(t)
		collsMock.AssertExpectations(t)
	})

	//t.Run("Can't get task config, storage returns error", func(t *testing.T) {
	//	collsMock := &collectionMock.Collections{}
	//	for _, collection := range getCollections() {
	//		collsMock.On("Get", mock.Anything, spaceID, envID, collection.ID).
	//			Return(collection, nil).
	//			Once()
	//	}
	//
	//	rMock := &rolesMock.Roles{}
	//	for _, role := range getRoles() {
	//		rMock.On("Get", mock.Anything, spaceID, role.ID).
	//			Return(role, nil).
	//			Once()
	//	}
	//
	//	clMock := &clientsMock.Clients{}
	//	for _, client := range getClients() {
	//		clMock.On("Get", mock.Anything, spaceID, client.ID).
	//			Return(client, nil).
	//			Once()
	//	}
	//
	//	itmMock := &itemsMock.Items{}
	//	for _, act := range getActions() {
	//		itmMock.On("Get", mock.Anything, spaceID, envID, extension.ActionsCollectionID, act.ID).
	//			Return(act, nil).
	//			Once()
	//	}
	//
	//	itmMock.On(
	//		"Find", mock.Anything, spaceID, envID, tasks.TaskConfigCollection,
	//		mock.Anything,
	//	).
	//		Return(nil, 0, errors.New("can't get task configs")).
	//		Once()
	//
	//	setup := newSetup(&content.Content{
	//		Roles:       rMock,
	//		Clients:     clMock,
	//		Collections: collsMock,
	//		Items:       itmMock,
	//	}, t)
	//
	//	err := setup.Check(context.Background())
	//
	//	require.Error(t, err)
	//	assert.ErrorContains(t, err, "task configs check error")
	//
	//	rMock.AssertExpectations(t)
	//	clMock.AssertExpectations(t)
	//	collsMock.AssertExpectations(t)
	//})
}
