package setup

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/content"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
	rolesMock "git.perx.ru/perxis/perxis-go/pkg/roles/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestSetup_InstallRoles(t *testing.T) {

	tests := []struct {
		name      string
		roles     []*roles.Role
		rolesCall func(svc *rolesMock.Roles)
		wantErr   func(t *testing.T, err error)
	}{
		{
			name:      "Nil role",
			roles:     nil,
			rolesCall: func(svc *rolesMock.Roles) {},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:  "Install one role success",
			roles: []*roles.Role{{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}},
			rolesCall: func(svc *rolesMock.Roles) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}).Return(&roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}, nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:  "Install one role fails",
			roles: []*roles.Role{{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}},
			rolesCall: func(svc *rolesMock.Roles) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("some error")).Once()
				svc.On("Create", mock.Anything, &roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}).Return(nil, errors.New("failed to install")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install role: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при настройке роли 1(test)")
			},
		},
		{
			name: "Install multiple roles success",
			roles: []*roles.Role{
				{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"},
				{ID: "2", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}},
			rolesCall: func(svc *rolesMock.Roles) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Get", mock.Anything, "sp", "2").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}).Return(&roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}, nil).Once()
				svc.On("Create", mock.Anything, &roles.Role{ID: "2", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}).Return(&roles.Role{ID: "2", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}, nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:  "Install multiple roles fails",
			roles: []*roles.Role{{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}, {ID: "2", SpaceID: "sp", Environments: []string{"env"}}},
			rolesCall: func(svc *rolesMock.Roles) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("some error")).Once()
				svc.On("Get", mock.Anything, "sp", "2").Return(nil, errors.New("some error")).Once()
				svc.On("Create", mock.Anything, &roles.Role{ID: "1", SpaceID: "sp", Environments: []string{"env"}, Description: "test"}).Return(nil, errors.New("failed to install")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install role: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при настройке роли 1(test)")
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			r := &rolesMock.Roles{}
			if tt.rolesCall != nil {
				tt.rolesCall(r)
			}

			s := NewSetup(&content.Content{Roles: r}, "sp", "env", nil)
			s.AddRoles(tt.roles)
			tt.wantErr(t, s.InstallRoles(context.Background()))
		})
	}
}
