package setup

import (
	"context"
	"reflect"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/collections"
	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/extension"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"go.uber.org/zap"
)

var (
	ErrCheckCollections     = errors.New("collections check error")
	ErrInstallCollections   = errors.New("failed to install collections")
	ErrUninstallCollections = errors.New("failed to uninstall collections")
)

type CollectionsOption func(c *CollectionConfig)
type UpdateCollectionFn func(s *Setup, exist, new *collections.Collection) (coll *collections.Collection, upd bool, setSchema bool, err error)
type DeleteCollectionFn func(s *Setup, col *collections.Collection) (bool, error)

type CollectionConfig struct {
	collection *collections.Collection
	UpdateFn   UpdateCollectionFn
	DeleteFn   DeleteCollectionFn
}

func NewCollectionConfig(collection *collections.Collection, opt ...CollectionsOption) CollectionConfig {
	c := CollectionConfig{collection: collection}

	DefaultUpdateCollectionStrategy()(&c)
	DeleteCollectionIfRemove()(&c)

	for _, o := range opt {
		o(&c)
	}

	return c
}

func OverwriteCollection() CollectionsOption {
	return func(c *CollectionConfig) {
		c.UpdateFn = func(s *Setup, old, new *collections.Collection) (*collections.Collection, bool, bool, error) {
			return new, true, true, nil
		}
	}
}

func KeepExistingCollection() CollectionsOption {
	return func(c *CollectionConfig) {
		c.UpdateFn = func(s *Setup, old, new *collections.Collection) (*collections.Collection, bool, bool, error) {
			return old, false, false, nil
		}
	}
}

func DeleteCollection() CollectionsOption {
	return func(c *CollectionConfig) {
		c.DeleteFn = func(s *Setup, collection *collections.Collection) (bool, error) { return true, nil }
	}
}

func DeleteCollectionIfRemove() CollectionsOption {
	return func(c *CollectionConfig) {
		c.DeleteFn = func(s *Setup, collection *collections.Collection) (bool, error) { return s.IsRemove(), nil }
	}
}

func isMetadataExtensionEqual(s1, s2 *schema.Schema) bool {
	if s1.Metadata == nil && s2.Metadata == nil {
		return true
	}

	if s1.Metadata == nil || s2.Metadata == nil {
		return false
	}

	return s1.Metadata[extension.ExtensionMetadataKey] == s2.Metadata[extension.ExtensionMetadataKey]
}

func DefaultUpdateCollectionStrategy() CollectionsOption {
	return func(c *CollectionConfig) {
		c.UpdateFn = func(s *Setup, exist, collection *collections.Collection) (*collections.Collection, bool, bool, error) {
			if !s.IsForce() && !collection.IsView() && !exist.IsView() && !isMetadataExtensionEqual(collection.Schema, exist.Schema) {
				return nil, false, false, errors.WithDetailf(collections.ErrAlreadyExists, "Коллекция с идентификатором '%s' "+
					"уже существует. Удалите ее или вызовите установку расширения с флагом Force", collection.ID)
			}

			if len(exist.Tags) > 0 {
				collection.Tags = data.SetFromSlice(append(exist.Tags, collection.Tags...))
			}

			var update, setSchema bool
			update = collection.Name != exist.Name || collection.IsSingle() != exist.IsSingle() || collection.IsSystem() != exist.IsSystem() ||
				collection.IsNoData() != exist.IsNoData() || collection.Hidden != exist.Hidden || collection.IsView() != exist.IsView() && data.ElementsMatch(exist.Tags, collection.Tags)

			if exist.View != nil && collection.View != nil {
				update = update && *exist.View == *collection.View
			}

			setSchema = !collection.IsView() && !reflect.DeepEqual(exist.Schema, collection.Schema)

			return collection, update, setSchema, nil
		}
	}
}

func (s *Setup) InstallCollections(ctx context.Context) (err error) {
	if len(s.Collections) == 0 {
		return nil
	}

	s.logger.Debug("Install collections", zap.Int("Collections", len(s.Collections)))

	var migrate, setSchema bool

	for _, c := range s.Collections {
		setSchema, err = s.InstallCollection(ctx, c)
		if err != nil {
			s.logger.Error("Failed to install collection",
				zap.String("Collection ID", c.collection.ID),
				zap.String("Collection Name", c.collection.Name),
				zap.Error(err),
			)
			return errors.WithDetailf(errors.Wrap(err, "failed to install collection"), "Возникла ошибка при настройке коллекции %s(%s)", c.collection.Name, c.collection.ID)
		}
		if setSchema {
			migrate = true
		}
	}

	if migrate {
		if err = s.content.Environments.Migrate(ctx, s.SpaceID, s.EnvironmentID); err != nil {
			s.logger.Error(
				"Failed to migrate environment",
				zap.String("Space ID", s.SpaceID),
				zap.String("Environment ID", s.EnvironmentID),
				zap.Error(err),
			)

			return errors.WithDetail(errors.Wrap(err, "migrate"), "Возникла ошибка при миграции данных")
		}
	}

	return nil
}

func (s *Setup) InstallCollection(ctx context.Context, c CollectionConfig) (setSchema bool, err error) {
	collection := c.collection
	collection.SpaceID, collection.EnvID = s.SpaceID, s.EnvironmentID

	var exist *collections.Collection
	// isForce - не удалять коллекцию, если она уже существует
	exist, err = s.content.Collections.Get(ctx, collection.SpaceID, collection.EnvID, collection.ID)
	if err != nil && !strings.Contains(err.Error(), collections.ErrNotFound.Error()) {
		return false, err
	}

	if exist == nil {
		setSchema = !collection.IsView()
		_, err = s.content.Collections.Create(ctx, collection)
		if err != nil {
			return false, err
		}
	} else {
		var upd bool
		collection, upd, setSchema, err = c.UpdateFn(s, exist, c.collection)
		if err != nil {
			return false, err
		}
		if upd {
			if err = s.content.Collections.Update(ctx, collection); err != nil {
				return false, err
			}
		}
	}

	if setSchema {
		err = s.content.Collections.SetSchema(ctx, collection.SpaceID, collection.EnvID, collection.ID, collection.Schema)
		if err != nil {
			return false, err
		}
	}

	return setSchema, nil
}

func (s *Setup) CheckCollections(ctx context.Context) error {
	if len(s.Collections) == 0 {
		return nil
	}

	s.logger.Debug("Check collections", zap.Int("Collections", len(s.Collections)))

	var errs []error
	for _, c := range s.Collections {
		if err := s.CheckCollection(ctx, c); err != nil {
			errs = append(errs, errors.WithDetailf(err, "Не найдена коллекция %s(%s)", c.collection.ID, c.collection.Name))
		}
	}

	if len(errs) > 0 {
		return errors.WithErrors(ErrCheckCollections, errs...)
	}

	return nil
}

func (s *Setup) CheckCollection(ctx context.Context, c CollectionConfig) (err error) {
	_, err = s.content.Collections.Get(ctx, s.SpaceID, s.EnvironmentID, c.collection.ID)
	return err
}

func (s *Setup) UninstallCollections(ctx context.Context) error {
	if len(s.Collections) == 0 {
		return nil
	}

	s.logger.Debug("Uninstall collections", zap.Int("Collections", len(s.Collections)))

	for _, c := range s.Collections {
		if err := s.UninstallCollection(ctx, c); err != nil {
			s.logger.Error("Failed to uninstall collection",
				zap.String("Collection ID", c.collection.ID),
				zap.String("Collection Name", c.collection.Name),
				zap.Error(err),
			)
			return errors.WithDetailf(errors.Wrap(err, "failed to uninstall collection"), "Возникла ошибка при удалении коллекции %s(%s)", c.collection.Name, c.collection.ID)
		}
	}

	return nil
}

func (s *Setup) UninstallCollection(ctx context.Context, c CollectionConfig) error {
	ok, err := c.DeleteFn(s, c.collection)
	if err != nil {
		return err
	}
	if ok {
		if err = s.content.Collections.Delete(ctx, s.SpaceID, s.EnvironmentID, c.collection.ID); err != nil && !strings.Contains(err.Error(), collections.ErrNotFound.Error()) {
			return err
		}
		s.removeItems(c.collection.ID) // после удаления коллекции нет смысла удалять ее элементы
	}
	return nil
}
