package setup

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/clients"
	clientsMock "git.perx.ru/perxis/perxis-go/pkg/clients/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/content"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestSetup_InstallClients(t *testing.T) {

	tests := []struct {
		name        string
		clients     []*clients.Client
		clientsCall func(svc *clientsMock.Clients)
		wantErr     func(t *testing.T, err error)
	}{
		{
			name:        "Nil clients",
			clients:     nil,
			clientsCall: func(svc *clientsMock.Clients) {},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:    "Install one client success",
			clients: []*clients.Client{{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}},
			clientsCall: func(svc *clientsMock.Clients) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}).Return(&clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}, nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:    "Install one client fails",
			clients: []*clients.Client{{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}},
			clientsCall: func(svc *clientsMock.Clients) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}).Return(nil, errors.New("some error")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install client: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при настройке клиента client(1)")
			},
		},
		{
			name:    "Install multiple clients success",
			clients: []*clients.Client{{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}, {ID: "2", SpaceID: "sp", Name: "client", RoleID: "role-2"}},
			clientsCall: func(svc *clientsMock.Clients) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Get", mock.Anything, "sp", "2").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}).Return(&clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}, nil).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "2", SpaceID: "sp", Name: "client", RoleID: "role-2"}).Return(&clients.Client{ID: "2", SpaceID: "sp", Name: "client", RoleID: "role-2"}, nil).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.NoError(t, err)
			},
		},
		{
			name:    "Install multiple clients fails",
			clients: []*clients.Client{{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}, {ID: "2", SpaceID: "sp", Name: "client", RoleID: "role-2"}},
			clientsCall: func(svc *clientsMock.Clients) {
				svc.On("Get", mock.Anything, "sp", "1").Return(nil, errors.New("not found")).Once()
				svc.On("Get", mock.Anything, "sp", "2").Return(nil, errors.New("not found")).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "1", SpaceID: "sp", Name: "client", RoleID: "role-1"}).Return(nil, errors.New("some error")).Once()
				svc.On("Create", mock.Anything, &clients.Client{ID: "2", SpaceID: "sp", Name: "client", RoleID: "role-2"}).Return(nil, errors.New("some error")).Once()
			},
			wantErr: func(t *testing.T, err error) {
				assert.Error(t, err)
				assert.EqualError(t, err, "failed to install client: some error")
				assert.Contains(t, errors.GetDetail(err), "Возникла ошибка при настройке клиента client(1)")
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			c := &clientsMock.Clients{}
			if tt.clientsCall != nil {
				tt.clientsCall(c)
			}

			s := NewSetup(&content.Content{Clients: c}, "sp", "env", nil)
			s.AddClients(tt.clients)
			tt.wantErr(t, s.InstallClients(context.Background()))
		})
	}
}
