package setup

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/clients"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"go.uber.org/zap"
)

var (
	ErrCheckClients     = errors.New("clients check error")
	ErrInstallClients   = errors.New("failed to install clients")
	ErrUninstallClients = errors.New("failed to uninstall clients")
)

type ClientsOption func(c *ClientConfig)
type UpdateClientFn func(s *Setup, exist, new *clients.Client) (*clients.Client, bool)
type DeleteClientFn func(s *Setup, client *clients.Client) bool

type ClientConfig struct {
	client   *clients.Client
	UpdateFn UpdateClientFn
	DeleteFn DeleteClientFn
}

func NewClientConfig(client *clients.Client, opt ...ClientsOption) ClientConfig {
	c := ClientConfig{client: client}

	UpdateExistingClient()(&c)
	DeleteClientIfRemove()(&c)

	for _, o := range opt {
		o(&c)
	}

	return c
}

func OverwriteClient() ClientsOption {
	return func(c *ClientConfig) {
		c.UpdateFn = func(s *Setup, old, new *clients.Client) (*clients.Client, bool) { return new, true }
	}
}

func KeepExistingClient() ClientsOption {
	return func(c *ClientConfig) {
		c.UpdateFn = func(s *Setup, old, new *clients.Client) (*clients.Client, bool) { return old, false }
	}
}

func DeleteClient() ClientsOption {
	return func(c *ClientConfig) {
		c.DeleteFn = func(s *Setup, client *clients.Client) bool { return true }
	}
}

func DeleteClientIfRemove() ClientsOption {
	return func(c *ClientConfig) {
		c.DeleteFn = func(s *Setup, client *clients.Client) bool { return s.IsRemove() }
	}
}

func UpdateExistingClient() ClientsOption {
	return func(c *ClientConfig) {
		c.UpdateFn = func(s *Setup, exist, client *clients.Client) (*clients.Client, bool) {
			if exist.Name == "" {
				exist.Name = client.Name
			}

			if exist.Description == "" {
				exist.Description = client.Description
			}

			if exist.OAuth == nil {
				exist.OAuth = client.OAuth
			}

			if exist.TLS == nil {
				exist.TLS = client.TLS
			}

			if exist.APIKey == nil {
				exist.APIKey = client.APIKey
			}

			exist.Disabled = client.Disabled
			exist.RoleID = client.RoleID

			return exist, true
		}
	}
}

func (s *Setup) InstallClients(ctx context.Context) error {
	if len(s.Clients) == 0 {
		return nil
	}

	s.logger.Debug("Install clients", zap.String("Space ID", s.SpaceID))

	for _, c := range s.Clients {
		err := s.InstallClient(ctx, c)
		if err != nil {
			s.logger.Error("Failed to install client", zap.String("Client ID", c.client.ID), zap.String("Client Name", c.client.Name), zap.Error(err))
			return errors.WithDetailf(errors.Wrap(err, "failed to install client"), "Возникла ошибка при настройке клиента %s(%s)", c.client.Name, c.client.ID)
		}
	}
	return nil
}

func (s *Setup) InstallClient(ctx context.Context, c ClientConfig) error {
	client := c.client
	client.SpaceID = s.SpaceID

	if s.IsForce() {
		s.content.Clients.Delete(ctx, s.SpaceID, c.client.ID)
		_, err := s.content.Clients.Create(ctx, c.client)
		return err
	}

	exist, err := s.content.Clients.Get(ctx, s.SpaceID, c.client.ID)
	if err != nil {
		if !strings.Contains(err.Error(), clients.ErrNotFound.Error()) {
			return err
		}

		_, err = s.content.Clients.Create(ctx, c.client)
		return err
	}

	if client, upd := c.UpdateFn(s, exist, c.client); upd {
		return s.content.Clients.Update(ctx, client)
	}

	return nil
}

func (s *Setup) CheckClients(ctx context.Context) (err error) {
	if len(s.Clients) == 0 {
		return nil
	}

	var errs []error
	s.logger.Debug("Check clients", zap.String("Space ID", s.SpaceID))
	for _, c := range s.Clients {
		err := s.CheckClient(ctx, c.client)
		if err != nil {
			errs = append(errs, errors.WithDetailf(err, "Не найден клиент %s(%s)", c.client.Name, c.client.ID))
		}
	}

	if len(errs) > 0 {
		return errors.WithErrors(ErrCheckClients, errs...)
	}

	return nil
}

func (s *Setup) CheckClient(ctx context.Context, client *clients.Client) error {
	_, err := s.content.Clients.Get(ctx, s.SpaceID, client.ID)
	return err
}

func (s *Setup) UninstallClients(ctx context.Context) error {
	if len(s.Clients) == 0 {
		return nil
	}

	s.logger.Debug("Uninstall clients", zap.String("Space ID", s.SpaceID))

	for _, c := range s.Clients {
		if err := s.UninstallClient(ctx, c); err != nil {
			s.logger.Error("Failed to uninstall client", zap.String("Client ID", c.client.ID), zap.String("Client Name", c.client.Name), zap.Error(err))
			return errors.WithDetailf(errors.Wrap(err, "failed to uninstall client"), "Возникла ошибка при удалении клиента %s(%s)", c.client.Name, c.client.ID)
		}
	}

	return nil
}

func (s *Setup) UninstallClient(ctx context.Context, c ClientConfig) error {
	if c.DeleteFn(s, c.client) {
		if err := s.content.Clients.Delete(ctx, s.SpaceID, c.client.ID); err != nil && !strings.Contains(err.Error(), clients.ErrNotFound.Error()) {
			return err
		}
	}
	return nil
}
