package walk

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestWalker_DataWalk(t *testing.T) {
	s := schema.New(
		"a", field.String(),
		"b", field.String(),
		"obj1", field.Object(
			"a", field.String(),
			"b", field.String(),
			"obj2", field.Object(
				"a", field.String(),
				"b", field.String(),
			),
			"obj3", field.Object(
				"e", field.String(),
			),
		),
		"slice", field.Array(field.String()),
		"inline_outer_str", field.Object(
			true,
			"inline_str_1", field.String(),
			"inline_obj", field.Object(
				true,
				"inline_str_2", field.String(),
			),
		),
	)

	tests := []struct {
		name        string
		config      *WalkConfig
		src         map[string]interface{}
		dst         map[string]interface{}
		res         map[string]interface{}
		wantChanged bool
		wantErr     bool
	}{
		{"generic",
			&WalkConfig{
				Fields: map[string]FieldConfig{
					"obj1.a":       {Fn: KeepSrc},
					"slice.1":      {Fn: KeepSrc},
					"inline_str_1": {Fn: KeepSrc},
					"inline_str_2": {Fn: KeepSrc},
				},
			},
			map[string]interface{}{
				"a": "src_a",
				"b": "src_b",
				"obj1": map[string]interface{}{
					"a": "src_obj1_a",
					"b": "src_obj1_b",
					"obj2": map[string]interface{}{
						"a": "dst_obj1_obj2_a",
					},
					"obj3": map[string]interface{}{
						"e": "dst_obj1_obj3_e",
					},
				},
				"inline_str_1": "src_inline_1",
				"inline_str_2": "src_inline_2",
				"slice":        []interface{}{"src_s1", "src_s2"},
			},
			map[string]interface{}{
				"a":                 "dst_a",
				"field_not_extists": "remove",
				"obj1": map[string]interface{}{
					"a": "dst_obj1_a",
					"obj2": map[string]interface{}{
						"a": "dst_obj1_obj2_a",
					},
				},
				"inline_str_1": "dst_inline_1",
				"inline_str_2": "dst_inline_2",
				"slice":        []interface{}{"dst_s1", "dst_s2", "dst_s3"},
			},
			map[string]interface{}{
				"a": "dst_a",
				"obj1": map[string]interface{}{
					"a": "src_obj1_a",
					"obj2": map[string]interface{}{
						"a": "dst_obj1_obj2_a",
					},
				},
				"inline_str_1": "src_inline_1",
				"inline_str_2": "src_inline_2",
				"slice":        []interface{}{"dst_s1", "src_s2", "dst_s3"},
			},
			false, false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			m := NewWalker(s, tt.config)
			dst := tt.dst
			res, _, err := m.DataWalk(context.Background(), dst, tt.src)
			assert.Equal(t, tt.res, res)
			if tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
