package validate

import (
	"context"
	"sort"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/expr"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
)

const ValidatorPriority = 2000

type Validator interface {
	Validate(ctx context.Context, f *field.Field, v interface{}) error
}

type Validators []Validator

func (l Validators) Len() int { return len(l) }
func (l Validators) Less(i, j int) bool {
	pi, pj := ValidatorPriority, ValidatorPriority
	if o, ok := l[i].(field.PriorityOption); ok {
		pi = o.GetPriority()
	}
	if o, ok := l[j].(field.PriorityOption); ok {
		pj = o.GetPriority()
	}
	if pi == pj {
		return field.GetOptionName(l[i]) < field.GetOptionName(l[j])
	}
	return pi < pj
}
func (l Validators) Swap(i, j int) { l[i], l[j] = l[j], l[i] }

func getValidators(f *field.Field) Validators {
	var vs Validators
	for _, o := range f.Options {
		if v, ok := o.(Validator); ok {
			vs = append(vs, v)
		}
	}
	sort.Sort(vs)
	return vs
}

func validateOptions(ctx context.Context, f *field.Field, v interface{}) error {
	var err error
	validators := getValidators(f)
	for _, i := range validators {
		err = i.Validate(ctx, f, v)
		if err != nil {
			return err
		}
	}
	return nil
}

func Validate(ctx context.Context, w field.Walker, v interface{}) error {
	if m, ok := v.(map[string]interface{}); ok {
		ctx = expr.WithEnv(ctx, m)
	}

	_, _, err := w.Walk(ctx, v, func(ctx context.Context, fld *field.Field, v interface{}) (res field.WalkFuncResult, err error) {
		enabled, _ := fld.IsEnabled(ctx)

		res.Value = v // Значение не меняется

		if !enabled {
			res.Stop = true
			return
		}

		if err = validateOptions(ctx, fld, v); err != nil {
			return
		}

		if validator, ok := fld.GetType().(Validator); ok {
			err = validator.Validate(ctx, fld, v)
		}

		return
	})

	return errors.Wrap(err, "validation error")
}

func init() {
	field.RegisterOption(minLength(0))
	field.RegisterOption(maxLength(0))
	field.RegisterOption(min(0))
	field.RegisterOption(max(0))
	field.RegisterOption(multipleOf(1))
	field.RegisterOption(enum{})
	field.RegisterOption(readonly(true))
	field.RegisterOption(required(true))
	field.RegisterOption(maxItems(0))
	field.RegisterOption(schema(true))
}
