package modify

import (
	"reflect"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestTrimSpace(t *testing.T) {
	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"String with spaces", field.String().AddOptions(TrimSpace()), "    string string   ", "string string", false},
		{"Nil", field.String().AddOptions(TrimSpace()), nil, nil, false},
		{"Spaces", field.String().AddOptions(TrimSpace()), "      ", "", false},
		{"Empty string", field.String().AddOptions(TrimSpace()), "", "", false},
		{"Not a string", field.String().AddOptions(TrimSpace()), 2, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, _, err := Modify(nil, tt.field, tt.data)
			if (err != nil) != tt.wantErr {
				t.Errorf("Modify() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Modify() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestModify(t *testing.T) {
	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		wantErr bool
		error   string
	}{
		{"String Length Max", field.String().AddOptions(TrimSpace()), " ", false, ""},
		{"String Length Min", field.String().AddOptions(TrimSpace()), " ", false, ""},
		{"Nil Length Max", field.String().AddOptions(TrimSpace()), nil, false, ""},
		{"Nil Length Min", field.String().AddOptions(TrimSpace()), nil, false, ""},
		{"Int Length Max", field.String().AddOptions(TrimSpace()), 1, true, "modification error: incorrect type: \"int\", expected \"string\""},
		{"Int Length Min", field.String().AddOptions(TrimSpace()), 1, true, "modification error: incorrect type: \"int\", expected \"string\""},
		{"Float Length Max", field.String().AddOptions(TrimSpace()), 1.0, true, "modification error: incorrect type: \"float64\", expected \"string\""},
		{"Float Length Min", field.String().AddOptions(TrimSpace()), 1.0, true, "modification error: incorrect type: \"float64\", expected \"string\""},
		{"Bool Length Max", field.String().AddOptions(TrimSpace()), true, true, "modification error: incorrect type: \"bool\", expected \"string\""},
		{"Bool Length Min", field.String().AddOptions(TrimSpace()), true, true, "modification error: incorrect type: \"bool\", expected \"string\""},
		{"Array Length Max", field.String().AddOptions(TrimSpace()), [1]string{""}, true, "modification error: incorrect type: \"array\", expected \"string\""},
		{"Array Length Min", field.String().AddOptions(TrimSpace()), [1]string{""}, true, "modification error: incorrect type: \"array\", expected \"string\""},
		{"Slice Length Max", field.String().AddOptions(TrimSpace()), []string{""}, true, "modification error: incorrect type: \"slice\", expected \"string\""},
		{"Slice Length Min", field.String().AddOptions(TrimSpace()), []string{""}, true, "modification error: incorrect type: \"slice\", expected \"string\""},
		{"Map Length Max", field.String().AddOptions(TrimSpace()), map[string]string{"": ""}, true, "modification error: incorrect type: \"map\", expected \"string\""},
		{"Map Length Min", field.String().AddOptions(TrimSpace()), map[string]string{"": ""}, true, "modification error: incorrect type: \"map\", expected \"string\""},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			_, _, err := Modify(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.error)
			}
			if !tt.wantErr {
				require.NoError(t, err)
			}
		})
	}
}
