package modify

import (
	"context"
	"sort"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/expr"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
)

const ModifierPriority = 1000

type Modifier interface {
	Modify(ctx context.Context, f *field.Field, v interface{}) (interface{}, bool, error)
}

type Modifiers []Modifier

func (l Modifiers) Len() int { return len(l) }
func (l Modifiers) Less(i, j int) bool {
	pi, pj := ModifierPriority, ModifierPriority
	if o, ok := l[i].(field.PriorityOption); ok {
		pi = o.GetPriority()
	}
	if o, ok := l[j].(field.PriorityOption); ok {
		pj = o.GetPriority()
	}
	if pi == pj {
		return field.GetOptionName(l[i]) < field.GetOptionName(l[j])
	}
	return pi < pj
}
func (l Modifiers) Swap(i, j int) { l[i], l[j] = l[j], l[i] }

func getModifiers(f *field.Field) Modifiers {
	var ms Modifiers
	for _, o := range f.Options {
		if m, ok := o.(Modifier); ok {
			ms = append(ms, m)
		}
	}
	sort.Sort(ms)
	return ms
}

func applyModifiers(ctx context.Context, f *field.Field, v interface{}) (interface{}, bool, error) {
	var err error
	var ok, modified bool
	modifiers := getModifiers(f)
	for _, i := range modifiers {
		v, ok, err = i.Modify(ctx, f, v)
		if err != nil {
			return nil, false, err
		}
		modified = modified || ok
	}
	return v, modified, nil
}

func Modify(ctx context.Context, w field.Walker, v interface{}) (interface{}, bool, error) {
	if m, ok := v.(map[string]interface{}); ok {
		ctx = expr.WithEnv(ctx, m)
	}

	v, c, err := w.Walk(ctx, v, func(ctx context.Context, fld *field.Field, v interface{}) (res field.WalkFuncResult, err error) {
		var vv interface{}
		var changed bool

		if vv, changed, err = applyModifiers(ctx, fld, v); err != nil {
			return
		}

		if changed {
			v = vv
		}

		if modifier, ok := fld.GetType().(Modifier); ok {
			vv, ch, err := modifier.Modify(ctx, fld, v)

			if err != nil {
				return res, err
			}

			if ch {
				v = vv
			}
		}

		res.Value = v
		res.Changed = changed
		return
	})

	if err != nil {
		return nil, false, errors.Wrap(err, "modification error")
	}

	return v, c, nil
}

func init() {
	field.RegisterOption(trimSpace(true))
	field.RegisterOption(defaultValue{})
	field.RegisterOption(value{})
}
