package modify

import (
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDefault(t *testing.T) {
	now := time.Now()
	_ = now
	w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")

	tests := []struct {
		name    string
		field   *field.Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"String value", field.String(Default("some")), "same", "same", false},
		{"String", field.String(Default("some")), nil, "some", false},
		{"Int", field.Number(field.NumberFormatInt, Default(2)), nil, int64(2), false},
		{"Time", field.Time(Default(now)), nil, now, false},
		{"Bool", field.Bool(Default(true)), nil, true, false},
		{"Object: nil with field default", field.Object("name", field.String(Default("test"))), nil, nil, false},
		{"Object: nil with object default", field.Object("name", field.String(Default("test"))).AddOptions(Default(map[string]interface{}{"name": "a"})), nil, map[string]interface{}{"name": "a"}, false},
		{"Object: empty", field.Object(
			"a", field.String(Default("a")),
			"b", field.Number(field.NumberFormatInt, Default(1)),
			"c", field.String(),
		),
			map[string]interface{}{},
			map[string]interface{}{"a": "a", "b": int64(1)},
			false},
		{"Array of time",
			field.Object("array", field.Array(field.Time())),
			map[string]interface{}{"array": []interface{}{"2012-11-01T22:08:41Z", "2012-11-01T22:08:41Z"}},
			map[string]interface{}{"array": []interface{}{w, w}},
			false},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := field.Decode(nil, tt.field, tt.data)
			require.NoError(t, err)
			got, _, err = Modify(nil, tt.field, got)
			if (err != nil) != tt.wantErr {
				t.Errorf("Modify() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			assert.Equal(t, tt.want, got)
		})
	}
}
