package field

import (
	"context"
	"fmt"
	"reflect"
	"time"

	"github.com/pkg/errors"
)

const DefaultTimeLayout = time.RFC3339

var timeType = &TimeType{}

type TimeParameters struct {
	Layout string `json:"layout,omitempty"`
}

func (p TimeParameters) Type() Type                  { return timeType }
func (p TimeParameters) Clone(reset bool) Parameters { return &p }

func (p TimeParameters) GetLayout() string {
	if p.Layout != "" {
		return p.Layout
	}
	return DefaultTimeLayout
}

type TimeType struct{}

func (TimeType) Name() string {
	return "time"
}

func (TimeType) NewParameters() Parameters {
	return &TimeParameters{}
}

func (TimeType) IsEmpty(v interface{}) bool {
	t, _ := v.(time.Time)
	return t.IsZero()
}

func (TimeType) Decode(_ context.Context, field *Field, v interface{}) (interface{}, error) {
	params, ok := field.Params.(*TimeParameters)
	if !ok {
		return nil, errors.New("TimeType: field type parameters required")
	}

	if v == nil {
		return v, nil
	}
	switch val := v.(type) {

	case string:
		if t, err := time.Parse(params.GetLayout(), val); err != nil {
			return nil, fmt.Errorf("TimeType: decode error %w", err)
		} else {
			return t, nil
		}
	case time.Time:
		return v, nil
	}
	return nil, fmt.Errorf("TimeType: decode: unsupported value type : \"%s\"", reflect.ValueOf(v).Kind())
}

func (TimeType) Encode(_ context.Context, field *Field, v interface{}) (interface{}, error) {
	params, ok := field.Params.(*TimeParameters)
	if !ok {
		return nil, errors.New("TimeType: field type parameters required")
	}

	if v == nil {
		return v, nil
	}
	if t, ok := v.(time.Time); ok {
		return t.Format(params.GetLayout()), nil
	}
	return nil, fmt.Errorf("TimeType: encode: unsupported value type : \"%s\"", reflect.ValueOf(v).Kind())
}

func Time(o ...interface{}) *Field {
	return NewField(&TimeParameters{}, o...)
}
