package field

import (
	"fmt"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestObjectField_Decode(t *testing.T) {
	w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{
			"Correct",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{"name": "string", "date": "2012-11-01T22:08:41Z", "bool": true},
			map[string]interface{}{"name": "string", "date": w, "bool": true},
			false,
		},
		{
			"Remove undefined fields",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{"name": "string", "date": "2012-11-01T22:08:41Z", "bool": true, "extra": "string"},
			map[string]interface{}{"name": "string", "date": w, "bool": true},
			false,
		},
		{
			"Empty data",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{},
			map[string]interface{}{},
			false,
		},
		{
			"Nil data",
			Object("name", String(), "date", Time(), "bool", Bool()),
			nil,
			nil,
			false,
		},
		{
			"Incorrect field",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{"name": "string", "date": "2012-11-01"},
			"decode error: 1 error occurred:\n\t* field 'date': TimeType: decode error parsing time \"2012-11-01\" as \"2006-01-02T15:04:05Z07:00\": cannot parse \"\" as \"T\"\n\n",
			true,
		},
		{
			"Incorrect type#1",
			Object("name", String(), "date", Time(), "bool", Bool()),
			[]interface{}{"name", "string", "date", "2012-11-01"},
			"decode error: incorrect type: \"slice\", expected \"map\"",
			true,
		},
		{
			"Incorrect type#2",
			Object("name", String(), "date", Time(), "bool", Bool()),
			"",
			"decode error: incorrect type: \"string\", expected \"map\"",
			true,
		},
		{
			"Incorrect type#3",
			Object("name", String(), "date", Time(), "bool", Bool()),
			"some",
			"decode error: incorrect type: \"string\", expected \"map\"",
			true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Decode(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.want.(string), fmt.Sprintf("Decode() error = %v, want %v", err, tt.want.(string)))
			}
			if !tt.wantErr {
				require.NoError(t, err)
				assert.Equal(t, got, tt.want, fmt.Sprintf("Decode() got = %v, want %v", got, tt.want))
			}
		})
	}
}

func TestObjectField_Encode(t *testing.T) {
	w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{
			"Correct",
			Object("name", String(), "date", Time(), "bool", Bool()),

			map[string]interface{}{"name": "string", "date": w, "bool": true},
			map[string]interface{}{"bool": true, "name": "string", "date": "2012-11-01T22:08:41Z"},
			false,
		},
		{
			"Additional properties",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{"name": "string", "date": w, "extra": "string", "bool": true},
			map[string]interface{}{"bool": true, "name": "string", "date": "2012-11-01T22:08:41Z"},
			false,
		},
		{
			"Empty data",
			Object("name", String(), "date", Time(), "bool", Bool()),
			map[string]interface{}{},
			map[string]interface{}{},
			false,
		},
		{
			"Nil data",
			Object("name", String(), "date", Time(), "bool", Bool()),
			nil,
			nil,
			false,
		},
		{
			"Incorrect type#1",
			Object("name", String(), "date", Time(), "bool", Bool()),
			[]interface{}{},
			"encode error: incorrect type: \"slice\", expected \"map\"",
			true,
		},
		{
			"Incorrect type#2",
			Object("name", String(), "date", Time(), "bool", Bool()),
			"",
			"encode error: incorrect type: \"string\", expected \"map\"",
			true,
		},
		{
			"Incorrect type#3",
			Object("name", String(), "date", Time(), "bool", Bool()),
			"some",
			"encode error: incorrect type: \"string\", expected \"map\"",
			true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Encode(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.want.(string), fmt.Sprintf("Encode() error = %v, want %v", err, tt.want.(string)))
			}
			if !tt.wantErr {
				require.NoError(t, err)
				assert.Equal(t, got, tt.want, fmt.Sprintf("Encode() got = %v, want %v", got, tt.want))
			}
		})
	}
}

func TestFieldNameValidate(t *testing.T) {
	tests := []struct {
		testName  string
		fieldName string
		wantErr   bool
	}{
		{
			"Correct field name",
			"name",
			false,
		},
		{
			"Not Latin",
			"название",
			true,
		},
		{
			"Start with a number",
			"1name",
			true,
		},
		{
			"Contains space",
			"field name",
			true,
		},
		{
			"Contains symbols",
			"name!",
			true,
		},
		{
			"Contains hyphen",
			"field-name",
			true,
		},
		{
			"Contains underscore (success)",
			"field_name",
			false,
		},
		{
			"Start with a capital letter (success)",
			"Name",
			false,
		},
		{
			"Contain a capital letter (success)",
			"fieldName",
			false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.testName, func(t *testing.T) {
			params := &ObjectParameters{Fields: map[string]*Field{tt.fieldName: String()}}
			err := objectType.ValidateParameters(params)
			if tt.wantErr {
				require.Error(t, err)
			} else {
				require.NoError(t, err)
			}
		})
	}
}
