package field

import (
	"context"
	"fmt"
	"reflect"
	"regexp"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/expr"
	"github.com/hashicorp/go-multierror"
)

var objectType = &ObjectType{}
var isValidName = regexp.MustCompile(`^[a-zA-Z][\w]*$`).MatchString

type ObjectParameters struct {
	Inline bool              `json:"inline"`
	Fields map[string]*Field `json:"fields"`
}

func (ObjectParameters) Type() Type { return objectType }

func (p ObjectParameters) Clone(reset bool) Parameters {
	if reset {
		p.Fields = nil
		return &p
	}

	flds := make(map[string]*Field)
	for k, v := range p.Fields {
		flds[k] = v.Clone(reset)
	}

	p.Fields = flds
	return &p
}

// IsInlineObject определяет являться ли поле name инлайн объектом
func (p ObjectParameters) IsInlineObject(name string) bool {
	fld, ok := p.Fields[name]
	if !ok {
		return false
	}

	if fldParams, ok := fld.Params.(*ObjectParameters); ok && fldParams.Inline {
		return true
	}

	return false
}

// GetFields возвращает поля объекта.
// Указание withInline позволяет так же включить поля указанные во вложенных inline объектам, и получиться поля для
// всех данных относящихся к текущему объекту.
func (p ObjectParameters) GetFields(withInline bool) map[string]*Field {
	fields := make(map[string]*Field)
	p.getFields(withInline, fields)
	return fields
}

func (p ObjectParameters) getFields(withInline bool, fields map[string]*Field) {
	for k, f := range p.Fields {
		if obj, ok := f.Params.(*ObjectParameters); ok && obj.Inline {
			obj.getFields(withInline, fields)
			continue
		}
		fields[k] = f
	}
}

func (p *ObjectParameters) Merge(parameters Parameters) error {
	op, ok := parameters.(*ObjectParameters)
	if !ok {
		return errors.New("invalid object parameters")
	}
	for k, fld := range op.Fields {
		if f, ok := p.Fields[k]; ok {
			if err := f.Merge(fld); err != nil {
				return err
			}
		} else {
			p.Fields[k] = fld
		}
	}
	return nil
}

type ObjectType struct{}

func (ObjectType) Name() string {
	return "object"
}

func (ObjectType) NewParameters() Parameters {
	return &ObjectParameters{}
}

func (ObjectType) IsEmpty(v interface{}) bool {
	m := reflect.ValueOf(v)
	return m.IsNil() || m.Len() == 0
}

type fieldNameCtx struct{}

var FieldName = fieldNameCtx{}

func (ObjectType) Walk(ctx context.Context, field *Field, v interface{}, fn WalkFunc, opts *WalkOptions) (interface{}, bool, error) {
	params, ok := field.Params.(*ObjectParameters)
	if !ok {
		return nil, false, errors.New("field parameters required")
	}

	// Объекта нет в данных, спускаться к полям мы не будем
	// Если необходимо что бы выполнялся Walk по полям необходимо передать пустой объект
	// Если нужно что бы всегда объект был, это можно сделать через Default
	if !opts.WalkSchema && v == nil {
		return nil, false, nil
	}

	m := reflect.ValueOf(v)

	if m.IsValid() {
		if m.Kind() != reflect.Map {
			return nil, false, errors.Errorf("incorrect type: \"%s\", expected \"map\"", m.Kind())
		}
	}

	if !opts.WalkSchema && m.IsNil() {
		return nil, false, nil
	}

	// Добавляем к переменным уровень объекта
	ctx = expr.WithEnvKV(ctx, "_", v)

	mapNew := make(map[string]interface{})

	var merr *multierror.Error
	var changed bool
	for name, fld := range params.Fields {
		ctxField := context.WithValue(ctx, FieldName, name)

		// Если поле является Inline-объектом, то передаются данные текущего объекта
		if p, ok := fld.Params.(*ObjectParameters); ok && p.Inline {
			valueNew, valueChanged, err := fld.Walk(ctxField, v, fn, WalkOpts(opts))

			if err != nil {
				merr = multierror.Append(merr, errors.WithField(err, name))
			}

			// Значение было изменено и оно не пустое (Inline объект не активен)
			if valueChanged && valueNew != nil {
				changed = true
			}

			if valueNew != nil {
				for n, v := range valueNew.(map[string]interface{}) {
					mapNew[n] = v
				}
			}
		} else {
			// Если значение нет, мы используем nil
			var value interface{}
			if m.IsValid() && !m.IsZero() && !m.IsNil() {
				fieldValue := m.MapIndex(reflect.ValueOf(name))
				if fieldValue.IsValid() {
					value = fieldValue.Interface()
				}
			}

			valueNew, valueChanged, err := fld.Walk(ctxField, value, fn, WalkOpts(opts))

			if err != nil {
				merr = multierror.Append(merr, errors.WithField(err, name))
			}

			// Если значение было изменено мы заменяем его на новое
			if valueChanged {
				changed = true
				value = valueNew
			}

			// Если значение не пустое, мы записываем поле в результат
			if value != nil {
				mapNew[name] = value
			}

		}
	}

	if merr != nil {
		//merr.ErrorFormat = func(i []error) string {
		//	return fmt.Sprintf("%d error(s)", len(i))
		//}
		return nil, false, merr
	}

	if v == nil || !m.IsValid() || m.IsZero() || m.IsNil() {
		return nil, false, nil
	}

	// Проверяем изменилось ли количество полей объекта.
	// Inline-объект игнорирует изменение количества полей, так как получает так же поля родительского объекта.
	if !changed && !params.Inline {
		changed = m.Len() != len(mapNew)
	}

	// Объект всегда возвращает новый модифицированный результат
	return mapNew, changed, nil
}

func (ObjectType) ValidateParameters(p Parameters) error {
	params, ok := p.(*ObjectParameters)
	if !ok {
		return nil
	}

	if len(params.Fields) > 0 {
		for k := range params.Fields {
			if !isValidName(k) {
				return fmt.Errorf("field name '%s' must be in Latin, must not start with a number, "+
					"must not contain spaces - only characters '_' can be used", k)
			}
		}
	}
	return nil
}

func Object(kv ...interface{}) *Field {
	params := &ObjectParameters{Fields: make(map[string]*Field)}

	if len(kv) > 0 {
		inline, ok := kv[0].(bool)
		if ok {
			params.Inline = inline
			kv = kv[1:]
		}
	}

	var order []interface{}
	i := 0
	for {
		if i+2 > len(kv) {
			break
		}
		k, v := kv[i], kv[i+1]
		name, kOk := k.(string)
		field, vOk := v.(*Field)
		if !kOk || !vOk {
			break
		}

		params.Fields[name] = field
		order = append(order, name)

		err := objectType.ValidateParameters(params)
		if err != nil {
			panic(err.Error())
		}

		i += 2
	}

	fld := NewField(params, kv[i:]...)
	if len(order) > 0 {
		fld = fld.WithUI(&UI{
			Options: map[string]interface{}{"fields": order},
		})
	}

	return fld
}

func AddField(field *Field, name string, fld *Field) error {
	switch params := field.Params.(type) {
	case *ObjectParameters:
		if params.Fields == nil {
			params.Fields = make(map[string]*Field)
		}
		params.Fields[name] = fld
	case *ArrayParameters:
		params.Item = fld
	default:
		return errors.New("AddField not supported")
	}

	return nil
}

func RemoveAllFields(obj *Field) error {
	params, ok := obj.Params.(*ObjectParameters)
	if !ok {
		return errors.New("obj is not an object")
	}
	params.Fields = make(map[string]*Field)
	return nil
}
