package field

import (
	"reflect"
	"testing"
)

func TestLocationField_Decode(t *testing.T) {
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"Correct",
			Location(),
			map[string]interface{}{
				"address":  "msk",
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}},
			},
			&GeoObject{"msk", &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{55.7042351, 37.6152822}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []float64{55.7042351, 37.6152822}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{55, 37}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55, 37}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{180, 90}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{180, 90}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{-180, -90}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{-180, -90}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []int{55, 37}},
			},
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55, 37}}},
			false},
		{"Correct",
			Location(),
			map[string]interface{}{
				"address": "msk",
			},
			&GeoObject{Address: "msk"},
			false},
		{"Correct", Location(), nil, nil, false},

		{"Wrong data", Location(), "", nil, true},
		{"Wrong data", Location(), []interface{}{"55.7042351", "37.6152822"}, nil, true},
		{"Wrong data", Location(), map[string]interface{}{"type": "Point", "coordinates": [][]interface{}{{55.7042351, 37.6152822}}}, nil, true},
		{"Wrong data", Location(), []interface{}{55.7042351, 37.6152822, 1.0}, nil, true},
		{"Wrong data",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []int{55, 37, 67}},
			},
			nil,
			true},
		{"Wrong data",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{180}},
			},
			nil,
			true},
		{"Wrong data",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{-180, -90.1}},
			},
			nil,
			true},
		{"Wrong data",
			Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"coordinates": []interface{}{180.1, 90.1}},
			},
			nil,
			true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Decode(nil, tt.field, tt.data)
			if (err != nil) != tt.wantErr {
				t.Errorf("Decode() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Decode() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestLocationField_Encode(t *testing.T) {
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"Correct", Location(),
			&GeoObject{Address: "msk", Geometry: &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			map[string]interface{}{
				"address":  "msk",
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}}},
			false},
		{"Correct", Location(),
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55.7042351, 37.6152822}}},
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}}},
			false},
		{"Correct", Location(),
			&GeoObject{Address: "msk"},
			map[string]interface{}{
				"address": "msk"},
			false},
		{"Correct", Location(),
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{55, 37}}},
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.0, 37.0}}},
			false},
		{"Correct", Location(),
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{180, 90}}},
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{180.0, 90.0}}},
			false},
		{"Correct", Location(),
			&GeoObject{Geometry: &GeoJSON{"Point", []float64{-180, -90}}},
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{-180.0, -90.0}}},
			false},
		{"Correct", Location(), nil, nil, false},
		{"Correct", Location(),
			&GeoObject{},
			map[string]interface{}{},
			false},

		{"Wrong data", Location(), "", nil, true},
		{"Wrong data", Location(), []interface{}{55.7042351, 37.6152822}, nil, true},
		{"Wrong data", Location(),
			map[string]interface{}{
				"address":  "msk",
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}}},
			nil,
			true},
		{"Wrong data", Location(),
			map[string]interface{}{
				"geometry": map[string]interface{}{"type": "Point", "coordinates": []interface{}{55.7042351, 37.6152822}}},
			nil,
			true},
		{"Wrong data", Location(),
			map[string]interface{}{
				"address": "msk"},
			nil,
			true},
		{"Wrong data", Location(), &GeoJSON{}, nil, true},
		{"Wrong data", Location(), &GeoJSON{Coordinates: []float64{55.7042351, 37.6152822}}, nil, true},
		{"Wrong data", Location(), &GeoObject{Geometry: &GeoJSON{"Point", []float64{-180, -90, 50}}}, nil, true},
		{"Wrong data", Location(), &GeoObject{Geometry: &GeoJSON{"Point", []float64{-180}}}, nil, true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Encode(nil, tt.field, tt.data)
			if (err != nil) != tt.wantErr {
				t.Errorf("Encode() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("Encode() got = %v, want %v", got, tt.want)
			}
		})
	}
}
