package field

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestBooleanField_Decode(t *testing.T) {
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"Correct data bool", Bool(), true, true, false},
		{"Correct data nil", Bool(), nil, nil, false},
		{"Wrong data int", Bool(), 1, "decode error: incorrect type: \"int\", expected \"boolean\"", true},
		{"Wrong data float", Bool(), 0.0, "decode error: incorrect type: \"float64\", expected \"boolean\"", true},
		{"Wrong data array", Bool(), [2]bool{true}, "decode error: incorrect type: \"array\", expected \"boolean\"", true},
		{"Wrong data slice", Bool(), []bool{true}, "decode error: incorrect type: \"slice\", expected \"boolean\"", true},
		{"Wrong data map", Bool(), map[bool]int{true: 1}, "decode error: incorrect type: \"map\", expected \"boolean\"", true},
		{"Wrong data string", Bool(), "2", "decode error: incorrect type: \"string\", expected \"boolean\"", true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Decode(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.want.(string), fmt.Sprintf("Decode() error = %v, want %v", err, tt.want.(string)))
			}
			if !tt.wantErr {
				require.NoError(t, err)
				assert.Equal(t, got, tt.want, fmt.Sprintf("Decode() got = %v, want %v", got, tt.want))
			}
		})
	}
}

func TestBooleanField_Encode(t *testing.T) {
	tests := []struct {
		name    string
		field   *Field
		data    interface{}
		want    interface{}
		wantErr bool
	}{
		{"Correct", Bool(), false, false, false},
		{"Correct data nil", Bool(), nil, nil, false},
		{"Wrong data int", Bool(), 1, "encode error: incorrect type: \"int\", expected \"boolean\"", true},
		{"Wrong data float", Bool(), 0.0, "encode error: incorrect type: \"float64\", expected \"boolean\"", true},
		{"Wrong data array", Bool(), [2]bool{true}, "encode error: incorrect type: \"array\", expected \"boolean\"", true},
		{"Wrong data slice", Bool(), []bool{true}, "encode error: incorrect type: \"slice\", expected \"boolean\"", true},
		{"Wrong data map", Bool(), map[bool]int{true: 1}, "encode error: incorrect type: \"map\", expected \"boolean\"", true},
		{"Wrong data string", Bool(), "2", "encode error: incorrect type: \"string\", expected \"boolean\"", true},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := Encode(nil, tt.field, tt.data)
			if tt.wantErr {
				require.Error(t, err)
				assert.EqualError(t, err, tt.want.(string), fmt.Sprintf("Encode() error = %v, want %v", err, tt.want.(string)))
			}
			if !tt.wantErr {
				require.NoError(t, err)
				assert.Equal(t, got, tt.want, fmt.Sprintf("Encode() got = %v, want %v", got, tt.want))
			}
		})
	}
}
