package roles

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/environments"
	mocksenvs "git.perx.ru/perxis/perxis-go/pkg/environments/mocks"
	"github.com/stretchr/testify/mock"
)

func TestRoleCanAccessEnvironment(t *testing.T) {
	type args struct {
		ctx      context.Context
		envscall func(envsservice *mocksenvs.Environments)
		role     *Role
		env      *environments.Environment
	}
	tests := []struct {
		name string
		args args
		want bool
	}{
		{
			name: "simple",
			args: args{
				ctx: context.Background(),
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"env1", "env2"},
				},
				env: &environments.Environment{ID: "env1", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test: e*",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"e*"},
				},
				env: &environments.Environment{ID: "env", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test: *n*",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"*n*"},
				},
				env: &environments.Environment{ID: "env", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test: *1",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"*1"},
				},
				env: &environments.Environment{ID: "env", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test (alias): ma*",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"ma*"},
				},
				env: &environments.Environment{ID: "env1", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test: *",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"*"},
				},
				env: &environments.Environment{ID: "env1", SpaceID: "sp"},
			},
			want: true,
		},
		{
			name: "glob env in role test: q*",
			args: args{
				ctx: context.Background(),
				envscall: func(envsservice *mocksenvs.Environments) {
					envsservice.On("Get", mock.Anything, mock.Anything, mock.Anything).Return(&environments.Environment{
						ID:      "env1",
						SpaceID: "space",
						Aliases: []string{"master"},
					}, nil).Once()
				},
				role: &Role{
					ID:           "1",
					SpaceID:      "space",
					Description:  "Current",
					Environments: []string{"q*"},
				},
				env: &environments.Environment{ID: "env1", SpaceID: "sp"},
			},
			want: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			envsservice := &mocksenvs.Environments{}
			if tt.args.envscall != nil {
				tt.args.envscall(envsservice)
			}

			if got := tt.args.role.CanAccessEnvironment(tt.args.ctx, tt.args.env, envsservice); got != tt.want {
				t.Errorf("hasEnvironmentAccess() = %v, want %v", got, tt.want)
			}
		})
	}
}
