package roles

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/data"
	"git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/permission"
)

const (
	AnonymousRole  = "anonymous"
	AuthorizedRole = "authorized"
	ViewRole       = "view"
)

type Role struct {
	// Внутренний идентификатор роли
	ID string `json:"id" bson:"_id"`

	// Идентификатор пространства
	SpaceID string `json:"spaceId" bson:"-"`

	// Описание роли, назначение
	Description string `json:"description" bson:"description"`

	// Список доступных окружений (ID или Alias)
	Environments []string `json:"environments" bson:"environments"`

	// Список правил доступа к коллекциям
	Rules permission.Rules `json:"rules" bson:"rules"`

	// Разрешить доступ API управления
	AllowManagement bool `json:"allow_management" bson:"allow_management"`
}

func (r Role) CanAccessEnvironment(ctx context.Context, env *environments.Environment, service environments.Environments) bool {
	if env.SpaceID == "" || env.ID == "" {
		return false
	}

	if r.AllowManagement {
		return true
	}

	// Если явно не указаны доступные окружения - доступ по умолчанию к окружению master
	if len(r.Environments) == 0 {
		r.Environments = []string{environments.DefaultEnvironment}
	}

	// Если окружение передано не полное, это означает, что надо его перезапросить
	if env.Description == "" && env.Aliases == nil && env.StateInfo == nil {
		if data.GlobMatch(env.ID, r.Environments...) {
			return true
		}

		var err error
		env, err = service.Get(ctx, env.SpaceID, env.ID)
		if err != nil || env == nil {
			return false
		}
	}

	aliases := append(env.Aliases, env.ID)

	for _, a := range aliases {
		if data.GlobMatch(a, r.Environments...) {
			return true
		}
	}

	return false
}
