package references

import (
	"git.perx.ru/perxis/perxis-go/pkg/items"
	pb "git.perx.ru/perxis/perxis-go/proto/references"
	"go.mongodb.org/mongo-driver/bson"
)

type Reference struct {
	ID           string `json:"id" bson:"id" mapstructure:"id"`
	CollectionID string `json:"collection_id" bson:"collection_id" mapstructure:"collection_id"`
	Disabled     bool   `json:"disabled,omitempty" bson:"disabled,omitempty" mapstructure:"disabled"`
}

func (r *Reference) MarshalBSON() ([]byte, error) {
	d := bson.D{
		{Key: "id", Value: r.ID},
		{Key: "collection_id", Value: r.CollectionID},
	}

	if r.Disabled {
		d = append(d, bson.E{Key: "disabled", Value: true})
	}

	return bson.Marshal(d)
}

func ReferenceFromPB(refPB *pb.Reference) *Reference {
	return &Reference{
		ID:           refPB.Id,
		CollectionID: refPB.CollectionId,
		Disabled:     refPB.Disabled,
	}
}

func ReferenceFromItem(item *items.Item) *Reference {
	if item == nil {
		return nil
	}

	return &Reference{
		ID:           item.ID,
		CollectionID: item.CollectionID,
	}
}

func ReferenceToPB(ref *Reference) *pb.Reference {
	return &pb.Reference{
		Id:           ref.ID,
		CollectionId: ref.CollectionID,
		Disabled:     ref.Disabled,
	}
}

func ReferenceListFromPB(listPB []*pb.Reference) []*Reference {
	list := make([]*Reference, 0, len(listPB))
	for _, refPB := range listPB {
		list = append(list, ReferenceFromPB(refPB))
	}
	return list
}

func (r *Reference) String() string {
	if r == nil {
		return ""
	}
	return r.CollectionID + "." + r.ID
}

func (r *Reference) Equal(r1 *Reference) bool {
	return r == r1 || r != nil && r1 != nil && r.CollectionID == r1.CollectionID && r.ID == r1.ID && r.Disabled == r1.Disabled
}

func EqualArrays(sr1, sr2 []*Reference) bool {
	if len(sr1) != len(sr2) {
		return false
	}
	for i, r := range sr1 {
		if !r.Equal(sr2[i]) {
			return false
		}
	}
	return true
}

func (r *Reference) IsValid() bool {
	return r != nil && r.ID != "" && r.CollectionID != "" && !r.Disabled
}

func (r *Reference) Fetch(i interface{}) interface{} {
	p, _ := i.(string)
	switch p {
	case "id":
		return r.ID
	case "collection_id":
		return r.CollectionID
	case "disabled":
		return r.Disabled
	default:
		panic("unknown parameter")
	}
}
