package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/organizations"
	mocksorgs "git.perx.ru/perxis/perxis-go/pkg/organizations/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestOrganizationsCache(t *testing.T) {

	const (
		orgId = "orgId"
		size  = 5
		ttl   = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		orgs := &mocksorgs.Organizations{}
		svc := CachingMiddleware(cache.NewCache(size, ttl))(orgs)

		orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "Organization"}, nil).Once()

		v1, err := svc.Get(ctx, orgId)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, orgId)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

		orgs.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Update", func(t *testing.T) {
			orgs := &mocksorgs.Organizations{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(orgs)

			orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "Organization"}, nil).Once()

			v1, err := svc.Get(ctx, orgId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			orgs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()
			err = svc.Update(ctx, &organizations.Organization{ID: orgId, Name: "OrganizationUPD"})

			orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "OrganizationUPD"}, nil).Once()

			v3, err := svc.Get(ctx, orgId)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается удаление объекта из кэша после обновления и получение заново из сервиса.")

			orgs.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			orgs := &mocksorgs.Organizations{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(orgs)

			orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "Organization"}, nil).Once()

			v1, err := svc.Get(ctx, orgId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			orgs.On("Delete", mock.Anything, mock.Anything).Return(nil).Once()
			err = svc.Delete(ctx, orgId)

			orgs.On("Get", mock.Anything, orgId).Return(nil, errNotFound).Once()

			_, err = svc.Get(ctx, orgId)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается удаление объекта из кэша после удаления из хранилища и получение ошибки от сервиса.")

		})

		t.Run("After TTL expired", func(t *testing.T) {
			orgs := &mocksorgs.Organizations{}
			svc := CachingMiddleware(cache.NewCache(size, ttl))(orgs)

			orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "Organization"}, nil).Once()

			v1, err := svc.Get(ctx, orgId)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, orgId)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			time.Sleep(2 * ttl)

			orgs.On("Get", mock.Anything, orgId).Return(&organizations.Organization{ID: orgId, Name: "Organization"}, nil).Once()

			v3, err := svc.Get(ctx, orgId)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается удаление объекта из кэша и получение заново из сервиса.")

			orgs.AssertExpectations(t)
		})
	})
}
