package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/locales"
	locmocks "git.perx.ru/perxis/perxis-go/pkg/locales/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestLocalesCache(t *testing.T) {

	const (
		loc1    = "loc1"
		loc2    = "loc2"
		spaceID = "spaceID"
		size    = 5
		ttl     = 20 * time.Millisecond
	)

	ctx := context.Background()

	t.Run("List from Cache", func(t *testing.T) {
		loc := &locmocks.Locales{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(loc)

		loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{{ID: loc1, Name: "name1", SpaceID: spaceID}}, nil).Once()

		vl1, err := svc.List(ctx, spaceID)
		require.NoError(t, err)

		vl2, err := svc.List(ctx, spaceID)
		require.NoError(t, err)
		assert.Same(t, vl1[0], vl2[0], "Ожидается что при повторном запросе объекты будут получены из кэша.")

		loc.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Delete", func(t *testing.T) {
			loc := &locmocks.Locales{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(loc)

			loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{{ID: loc1, Name: "name1", SpaceID: spaceID}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается что при повторном запросе объекты будут получены из кэша.")

			loc.On("Delete", mock.Anything, spaceID, loc1).Return(nil).Once()

			err = svc.Delete(ctx, spaceID, loc1)
			require.NoError(t, err)

			loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl3, 0, "Ожидается что после удаление объекты будут удалены из кеша.")

			loc.AssertExpectations(t)
		})

		t.Run("After Create", func(t *testing.T) {
			loc := &locmocks.Locales{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(loc)

			loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{{ID: loc1, Name: "name1", SpaceID: spaceID}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается что при повторном запросе объекты будут получены из кэша.")

			loc.On("Create", mock.Anything, mock.Anything).Return(&locales.Locale{ID: loc2, Name: "name2", SpaceID: spaceID}, nil).Once()

			_, err = svc.Create(ctx, &locales.Locale{ID: loc2, Name: "name2", SpaceID: spaceID})
			require.NoError(t, err)

			loc.On("List", mock.Anything, spaceID).
				Return([]*locales.Locale{
					{ID: loc1, Name: "name1", SpaceID: spaceID},
					{ID: loc2, Name: "name2", SpaceID: spaceID},
				}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl3, 2, "Ожидается что после создания нового объекта данные будут удалены из кеша и получены из сервиса.")

			loc.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			loc := &locmocks.Locales{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(loc)

			loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{{ID: loc1, Name: "name1", SpaceID: spaceID}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается что при повторном запросе объекты будут получены из кэша.")

			time.Sleep(2 * ttl)
			loc.On("List", mock.Anything, spaceID).Return([]*locales.Locale{{ID: loc1, Name: "name1", SpaceID: spaceID}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидается что элементы будут получены из кэша.")

			loc.AssertExpectations(t)
		})
	})
}
