package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/invitations"
	invmocks "git.perx.ru/perxis/perxis-go/pkg/invitations/mocks"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestLocalesCache(t *testing.T) {

	const (
		orgID = "orgID"
		email = "123@321.ru"
		invID = "invID"
		usrID = "usrID"
		size  = 5
		ttl   = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from Cache", func(t *testing.T) {
		inv := &invmocks.Invitations{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(inv)

		inv.On("Get", mock.Anything, invID).Return(&invitations.Invitation{ID: invID, Email: email, OrgID: orgID}, nil).Once()

		v1, err := svc.Get(ctx, invID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, invID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается что при повторном запросе объект будет получен из кэша.")

		inv.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("Get from Accept", func(t *testing.T) {
			inv := &invmocks.Invitations{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(inv)

			inv.On("Get", mock.Anything, invID).Return(&invitations.Invitation{ID: invID, Email: email, OrgID: orgID}, nil).Once()

			v1, err := svc.Get(ctx, invID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, invID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается что при повторном запросе объект будет получен из кэша.")

			inv.On("Accept", mock.Anything, invID, usrID).Return(nil).Once()
			inv.On("Get", mock.Anything, invID).Return(nil, errNotFound).Once()

			err = svc.Accept(ctx, invID, usrID)
			require.NoError(t, err)

			_, err = svc.Get(ctx, invID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после подтверждения объект будет удален из кэша и получена ошибка от сервиса.")

			inv.AssertExpectations(t)
		})

		t.Run("Get from Delete", func(t *testing.T) {
			inv := &invmocks.Invitations{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(inv)

			inv.On("Get", mock.Anything, invID).Return(&invitations.Invitation{ID: invID, Email: email, OrgID: orgID}, nil).Once()

			v1, err := svc.Get(ctx, invID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, invID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается что при повторном запросе объект будет получен из кэша.")

			inv.On("Delete", mock.Anything, invID).Return(nil).Once()
			inv.On("Get", mock.Anything, invID).Return(nil, errNotFound).Once()

			err = svc.Delete(ctx, invID)
			require.NoError(t, err)

			_, err = svc.Get(ctx, invID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидается что после удаления кэша будет очищен и получена ошибка от сервиса.")

			inv.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			inv := &invmocks.Invitations{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(inv)

			inv.On("Get", mock.Anything, invID).Return(&invitations.Invitation{ID: invID, Email: email, OrgID: orgID}, nil).Once()

			v1, err := svc.Get(ctx, invID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, invID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается что при повторном запросе объект будет получен из кэша.")

			time.Sleep(2 * ttl)

			inv.On("Get", mock.Anything, invID).Return(&invitations.Invitation{ID: invID, Email: email, OrgID: orgID}, nil).Once()

			v3, err := svc.Get(ctx, invID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидается что при истечении ttl кеш будет очищен..")

			inv.AssertExpectations(t)
		})
	})
}
