package id

import (
	"testing"

	"github.com/stretchr/testify/require"
)

func TestValidateID(t *testing.T) {
	tests := []struct {
		name    string
		id      string
		result  string
		wantErr bool
	}{
		{
			"Correct ID #1",
			"test",
			"test",
			false,
		},
		{
			"Correct ID #2",
			"test_1",
			"test_1",
			false,
		},
		{
			"Correct ID #3",
			"test_1-2",
			"test_1-2",
			false,
		},
		{
			"Trimmed ID #1",
			" test_1-2 ",
			"test_1-2",
			false,
		},
		{
			"Trimmed ID #2",
			"test_1-2 			",
			"test_1-2",
			false,
		},
		{
			"Fail, ID starts with number",
			"1test",
			"1test",
			true,
		},
		{
			"Fail, ID to long",
			"9QoulAlFbIcnQYLSudMistN1IczCWrXUTtN5EgNQJd516DN9UjXYJxieJ1RcsNcs1",
			"9QoulAlFbIcnQYLSudMistN1IczCWrXUTtN5EgNQJd516DN9UjXYJxieJ1RcsNcs1",
			true,
		},
		{
			"Fail, ID with symbols",
			"test!_)(&&",
			"test!_)(&&",
			true,
		},
		{
			"Fail, ID is empty",
			"   ",
			"",
			true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			res, err := ValidateID(tt.id)
			if (err != nil) != tt.wantErr {
				t.Errorf("ValidateID() error = %v, wantErr %v", err, tt.wantErr)
			}
			require.Equal(t, tt.result, res)
		})
	}
}

func TestValidateName(t *testing.T) {
	tests := []struct {
		nameTest string
		name     string
		result   string
		wantErr  bool
	}{
		{
			"Correct name #1",
			"test",
			"test",
			false,
		},
		{
			"Correct name #2",
			"test_1",
			"test_1",
			false,
		},
		{
			"Correct name #3",
			"test_1-2",
			"test_1-2",
			false,
		},
		{
			"Cyrillic name #1",
			"1ывлдаывоаждывоажыовалывоалджыварщшуджлкцталиамтвалытм тывжалдыжлдвттлвтмыдытфджтмфвмт",
			"1ывлдаывоаждывоажыовалывоалджыварщшуджлкцталиамтвалытм тывжалдыжлдвттлвтмыдытфджтмфвмт",
			false,
		},
		{
			"Cyrillic name #2",
			"Римский император Константин I Великий по достоинству оценил выгодное местоположение приморского Византия, расположенного на стыке Евро",
			"Римский император Константин I Великий по достоинству оценил выгодное местоположение приморского Византия, расположенного на стыке Евро",
			false,
		},
		{
			"Trimmed name #1",
			" test_1-3 ",
			"test_1-3",
			false,
		},
		{
			"Trimmed name #2",
			"test_1234 			",
			"test_1234",
			false,
		},
		{
			"Fail, name to long #1",
			"ChKRLdvWi0wYYPazuBXrIRtNFy96qGrhBDkuKQNd6N2DPV86IGdXVkeTjWj7qezKreIFUp9IUn03A8WJTTORHkgXAvkPuDVM8tVMcnHbR2hznooJ3gGUsXpn4uXo2QhsviHPyUKmE10GnkCOv9FgAMILNoFVHnIiSHI3cjWlGJglpS9YAMXFB1phOIRF5yol3jmPE7EeU1uZPUw9C2PChuksGsOuJQov07Zom0b13r6wOJv8PZVa4IKmjDDLGKlq1",
			"ChKRLdvWi0wYYPazuBXrIRtNFy96qGrhBDkuKQNd6N2DPV86IGdXVkeTjWj7qezKreIFUp9IUn03A8WJTTORHkgXAvkPuDVM8tVMcnHbR2hznooJ3gGUsXpn4uXo2QhsviHPyUKmE10GnkCOv9FgAMILNoFVHnIiSHI3cjWlGJglpS9YAMXFB1phOIRF5yol3jmPE7EeU1uZPUw9C2PChuksGsOuJQov07Zom0b13r6wOJv8PZVa4IKmjDDLGKlq1",
			true,
		},
		{
			"Fail, name to long #2",
			"Идентификаторы, различающиеся только регистром букв, являются различными. В языке существует ряд соглашений об использовании заглавных и строчных букв. В частности, в именах пакетов используются только строчные буквы. Все ключевые слова Go пишутся в нижнем регистре. Переменные, начинающиеся с заглавных букв, являются экспортируемыми (public), а начинающиеся со строчных букв — неэкспортируемыми (private).",
			"Идентификаторы, различающиеся только регистром букв, являются различными. В языке существует ряд соглашений об использовании заглавных и строчных букв. В частности, в именах пакетов используются только строчные буквы. Все ключевые слова Go пишутся в нижнем регистре. Переменные, начинающиеся с заглавных букв, являются экспортируемыми (public), а начинающиеся со строчных букв — неэкспортируемыми (private).",
			true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			res, err := ValidateName(tt.name)
			if (err != nil) != tt.wantErr {
				t.Errorf("ValidateName() error = %v, wantErr %v", err, tt.wantErr)
			}
			require.Equal(t, tt.result, res)
		})
	}
}
