package id

import (
	"regexp"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/rs/xid"
)

const (
	MaxLengthID   = 64
	MaxLengthName = 256
)

var (
	ErrEmptyID    = errors.New("empty id")
	ErrNotValidID = errors.New("id contains invalid characters")
	ErrLongID     = errors.Errorf("id too long (max %d symbols)", MaxLengthID)
	ErrEmptyName  = errors.New("empty name")
	ErrLongName   = errors.Errorf("name too long (max %d symbols)", MaxLengthName)
	isValidID     = regexp.MustCompile(`^[A-Za-z][A-Za-z0-9-_]*$`).MatchString
)

func GenerateNewID() string {
	return xid.New().String()
}

func ValidateID(id string) (string, error) {
	trimmedID := strings.TrimSpace(id)
	if trimmedID == "" {
		return trimmedID, ErrEmptyID
	}
	if len(trimmedID) > MaxLengthID {
		return trimmedID, ErrLongID
	}
	if !isValidID(trimmedID) {
		return trimmedID, errors.WithDetail(ErrNotValidID, "must begin with latin letters and contain latin letters, numbers or '-', '_'")
	}
	return trimmedID, nil
}

func ValidateName(name string) (string, error) {
	trimmedName := strings.TrimSpace(name)
	if trimmedName == "" {
		return trimmedName, ErrEmptyName
	}
	if len([]rune(trimmedName)) > MaxLengthName {
		return trimmedName, ErrLongName
	}
	return trimmedName, nil
}
