package filter

import (
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"git.perx.ru/perxis/perxis-go/pkg/schema/field"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.mongodb.org/mongo-driver/bson/primitive"
)

func TestFilterHandler(t *testing.T) {

	sch := schema.New(
		"str", field.String(),
		"num", field.Number(field.NumberFormatInt),
		"obj", field.Object(
			"bool", field.Bool(),
			"arr", field.Array(field.Time()),
			"list", field.Array(
				field.Object(
					"num1", field.Number(field.NumberFormatFloat),
					"str1", field.String(),
				),
			),
		),
		"date", field.Time(),
		"geo", field.Location(),
	)
	h := NewFilterHandler(sch)
	ph := NewFilterHandler(sch).SetTrimPrefix("data")

	h.SetQueryBuilder(NewMongoQueryBuilder())
	ph.SetQueryBuilder(NewMongoQueryBuilder())

	var err error

	t.Run("Validate", func(t *testing.T) {
		t.Run("Simple", func(t *testing.T) {
			t.Run("String", func(t *testing.T) {
				f := &Filter{Op: Equal, Field: "str", Value: "zzz"}
				err = h.Validate(f)
				require.NoError(t, err)

				f = &Filter{Op: Equal, Field: "data.str", Value: "zzz"}
				err = ph.Validate(f)
				require.NoError(t, err)
			})
			t.Run("Int", func(t *testing.T) {
				f := &Filter{Op: NotEqual, Field: "num", Value: 5.0}
				err = h.Validate(f)
				require.NoError(t, err)
				assert.IsType(t, int64(0), f.Value)

				f = &Filter{Op: NotEqual, Field: "data.num", Value: 5.0}
				err = ph.Validate(f)
				require.NoError(t, err)
				assert.IsType(t, int64(0), f.Value)
			})
			t.Run("Time", func(t *testing.T) {
				f := &Filter{Op: LessOrEqual, Field: "date", Value: "22 Dec 1997"}
				err = h.Validate(f)
				require.Error(t, err)

				f = &Filter{Op: LessOrEqual, Field: "data.date", Value: "22 Dec 1997"}
				err = ph.Validate(f)
				require.Error(t, err)
			})
			t.Run("Location", func(t *testing.T) {
				f := &Filter{Op: Near, Field: "geo", Value: ""}
				err = h.Validate(f)
				require.Error(t, err)

				f = &Filter{Op: Near, Field: "data.geo", Value: ""}
				err = ph.Validate(f)
				require.Error(t, err)

				fv := map[string]interface{}{
					"point":    []float64{55, 55},
					"distance": 1000,
				}

				f = &Filter{Op: Near, Field: "data.geo", Value: fv}
				err = ph.Validate(f)
				require.NoError(t, err)

				fv["distance"] = -1
				f = &Filter{Op: Near, Field: "data.geo", Value: fv}
				err = ph.Validate(f)
				require.Error(t, err)

			})
		})
		t.Run("Embedded array field", func(t *testing.T) {
			w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
			f := &Filter{Op: In, Field: "obj.arr", Value: []interface{}{"2012-11-01T22:08:41Z"}}
			err = h.Validate(f)
			require.NoError(t, err)
			assert.Equal(t, w, f.Value.([]interface{})[0])

			f = &Filter{Op: In, Field: "data.obj.arr", Value: []interface{}{"2012-11-01T22:08:41Z"}}
			err = ph.Validate(f)
			require.NoError(t, err)
			assert.Equal(t, w, f.Value.([]interface{})[0])
		})
		t.Run("Embedded string contains", func(t *testing.T) {
			f := &Filter{Op: Contains, Field: "obj.list.str1", Value: "zzz"}
			err = h.Validate(f)
			require.NoError(t, err)

			f = &Filter{Op: Contains, Field: "data.obj.list.str1", Value: "zzz"}
			err = ph.Validate(f)
			require.NoError(t, err)
		})
		t.Run("Compound filter with 'OR' operation", func(t *testing.T) {
			t.Run("No Err", func(t *testing.T) {
				w1, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
				w2, _ := time.Parse(time.RFC3339, "2015-12-01T22:08:41Z")

				ff := []*Filter{
					{Op: In, Field: "date", Value: []interface{}{"2012-11-01T22:08:41Z", "2015-12-01T22:08:41Z"}},
					{Op: Or, Field: "", Value: []*Filter{
						{Op: And, Field: "", Value: []*Filter{
							{Op: GreaterOrEqual, Field: "date", Value: "2012-11-01T22:08:41Z"},
							{Op: LessOrEqual, Field: "date", Value: "2015-12-01T22:08:41Z"},
						}},
						{Op: Equal, Field: "obj.bool", Value: true},
					}},
				}
				err = h.Validate(ff...)
				require.NoError(t, err)
				assert.ElementsMatch(t, []interface{}{w1, w2}, ff[0].Value.([]interface{}))
				assert.Equal(t, w1, ff[1].Value.([]*Filter)[0].Value.([]*Filter)[0].Value)
				assert.Equal(t, w2, ff[1].Value.([]*Filter)[0].Value.([]*Filter)[1].Value)

				ff = []*Filter{
					{Op: In, Field: "data.date", Value: []interface{}{"2012-11-01T22:08:41Z", "2015-12-01T22:08:41Z"}},
					{Op: Or, Field: "", Value: []*Filter{
						{Op: And, Field: "", Value: []*Filter{
							{Op: GreaterOrEqual, Field: "data.date", Value: "2012-11-01T22:08:41Z"},
							{Op: LessOrEqual, Field: "data.date", Value: "2015-12-01T22:08:41Z"},
						}},
						{Op: Equal, Field: "data.obj.bool", Value: true},
					}},
				}
				err = ph.Validate(ff...)
				require.NoError(t, err)
				assert.ElementsMatch(t, []interface{}{w1, w2}, ff[0].Value.([]interface{}))
				assert.Equal(t, w1, ff[1].Value.([]*Filter)[0].Value.([]*Filter)[0].Value)
				assert.Equal(t, w2, ff[1].Value.([]*Filter)[0].Value.([]*Filter)[1].Value)
			})
			t.Run("Multiple Errors", func(t *testing.T) {
				ff := []*Filter{
					{Op: In, Field: "date", Value: []interface{}{"5 Jan 2020", "10 June 2020"}},
					{Op: Or, Field: "", Value: []*Filter{
						{Op: And, Field: "", Value: []*Filter{
							{Op: GreaterOrEqual, Field: "date", Value: "2012-11-01T22:08:41Z"},
							{Op: LessOrEqual, Field: "date", Value: "2015-12-01T22:08:41Z"},
						}},
						{Op: Equal, Field: "obj.bool", Value: 15},
					}},
				}
				err = h.Validate(ff...)
				require.Error(t, err)
				assert.Equal(t, err.Error(), "2 validation error(s)")

				ff = []*Filter{
					{Op: In, Field: "data.date", Value: []interface{}{"5 Jan 2020", "10 June 2020"}},
					{Op: Or, Field: "", Value: []*Filter{
						{Op: And, Field: "", Value: []*Filter{
							{Op: GreaterOrEqual, Field: "data.date", Value: "2012-11-01T22:08:41Z"},
							{Op: LessOrEqual, Field: "data.date", Value: "2015-12-01T22:08:41Z"},
						}},
						{Op: Equal, Field: "data.obj.bool", Value: 15},
					}},
				}
				err = h.Validate(ff...)
				require.Error(t, err)
				assert.Equal(t, err.Error(), "2 validation error(s)")
			})
		})
	})

	t.Run("Build Query", func(t *testing.T) {
		t.Run("No Filters", func(t *testing.T) {
			res := h.Query()
			require.IsType(t, res, primitive.M{})

			pres := ph.Query()
			assert.Equal(t, res, pres, "пустой запрос с префиксом и без должны быть одинаковые")
		})
		t.Run("Equal String", func(t *testing.T) {
			f := &Filter{Op: Equal, Field: "data.str", Value: "zzz"}
			res := h.Query(f)
			b, ok := res.(primitive.M)
			require.True(t, ok)
			assert.Equal(t, primitive.M{"$and": primitive.A{primitive.M{"data.str": primitive.M{"$eq": "zzz"}}}}, b)

			pf := &Filter{Op: Equal, Field: "data.str", Value: "zzz"}
			pres := ph.Query(pf)
			assert.Equal(t, res, pres, "запрос в БД с полями с префиксом и без должны быть одинаковые")
		})
		t.Run("In Array", func(t *testing.T) {
			w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
			f := &Filter{Op: In, Field: "obj.arr", Value: []interface{}{w}}
			res := h.Query(f)
			b, ok := res.(primitive.M)
			require.True(t, ok)
			assert.Equal(t, primitive.M{"$and": primitive.A{primitive.M{"obj.arr": primitive.M{"$in": []interface{}{w}}}}}, b)
		})
		t.Run("Several ops for one field", func(t *testing.T) {
			w, _ := time.Parse(time.RFC3339, "2012-11-01T22:08:41Z")
			f := &Filter{Op: In, Field: "obj.arr", Value: []interface{}{w}}
			res := h.Query(f)
			b, ok := res.(primitive.M)
			require.True(t, ok)
			assert.Equal(t, primitive.M{"$and": primitive.A{primitive.M{"obj.arr": primitive.M{"$in": []interface{}{w}}}}}, b)
		})
	})
}

//func TestFilterHandler_Integration(t *testing.T) {
//	ctx := context.Background()
//
//	uri := os.Getenv("MONGO_URL")
//	if uri == "" {
//		uri = "mongodb://localhost:27017"
//	}
//	opts := options.Client().SetConnectTimeout(15 * time.Second).ApplyURI(uri)
//	client, err := mongo.Connect(context.Background(), opts)
//	require.NoError(t, err)
//	err = client.Ping(ctx, nil)
//	require.NoError(t, err)
//
//	sch := schema.New(
//		"name", field.String(validate.Required()),
//		"color", field.String(),
//		"qty", field.Number(field.NumberFormatInt),
//		"info", field.Object(
//			"is_fruit", field.Bool(),
//			"similar", field.Array(
//				field.Object(
//					"name", field.Number(field.NumberFormatFloat),
//					"color", field.String(),
//				),
//			),
//			"desc", field.String(),
//		),
//		"produced", field.Time(),
//		"shipment", field.Array(field.String()),
//	)
//
//	w1, _ := time.Parse(time.RFC3339, "2020-01-01T10:08:41Z")
//	w2, _ := time.Parse(time.RFC3339, "2020-05-01T10:08:41Z")
//	w3, _ := time.Parse(time.RFC3339, "2020-10-01T10:08:41Z")
//
//	items := []map[string]interface{}{
//		{
//			"name":  "apple",
//			"color": "red",
//			"qty":   25,
//			"info": map[string]interface{}{
//				"is_fruit": true,
//				"similar": []interface{}{
//					map[string]interface{}{"name": "pear", "color": "yellow"},
//					map[string]interface{}{"name": "lemon", "color": "yellow"},
//				},
//				"desc": "An apple is the edible fruit . Apple trees are cultivated worldwide and have religious and mythological " +
//					"significance in many cultures. Apples are eaten with honey at the Jewish New Year of Rosh Hashanah to symbolize a sweet new year.",
//			},
//			"produced":   w1,
//			"shipment":   []interface{}{"Russia", "Iran"},
//			"storepoint": map[string]interface{}{"type": "Point", "coordinates": []float64{55.751472, 37.618727}},
//		},
//		{
//			"name":  "orange",
//			"color": "orange",
//			"qty":   10,
//			"info": map[string]interface{}{
//				"is_fruit": true,
//				"similar": []interface{}{
//					map[string]interface{}{"name": "lemon", "color": "yellow"},
//					map[string]interface{}{"name": "grapefruit", "color": "red"},
//				},
//				"desc": "The orange is the edible fruit of various citrus species; a hybrid between pomelo and mandarin. Orange trees are widely grown" +
//					" in tropical and subtropical climates for their sweet fruit. The fruit of the orange tree can be eaten fresh, or processed for its juice or fragrant peel.",
//			},
//			"produced":   w2,
//			"shipment":   []interface{}{"Egypt", "Iran"},
//			"storepoint": map[string]interface{}{"type": "Point", "coordinates": []float64{55.716797, 37.552809}},
//		},
//		{
//			"name":  "tomato",
//			"color": "red",
//			"qty":   1,
//			"info": map[string]interface{}{
//				"is_fruit": false,
//				"similar": []interface{}{
//					map[string]interface{}{"name": "cucumber", "color": "green"},
//					map[string]interface{}{"name": "apple", "color": "yellow"},
//				},
//				"desc": "The tomato is the edible red berry. The tomato is consumed in diverse ways, raw or cooked, in many dishes, " +
//					"sauces, salads, and drinks. Numerous varieties of the tomato plant are widely grown in temperate climates across the world.",
//			},
//			"produced":   w3,
//			"shipment":   []interface{}{"Russia", "Italy"},
//			"storepoint": map[string]interface{}{"type": "Point", "coordinates": []float64{55.760688, 37.619125}},
//		},
//	}
//
//	db := client.Database("perxis_test_filter")
//	coll := db.Collection("items")
//	coll.Drop(ctx)
//
//	for _, item := range items {
//		_, err = coll.InsertOne(ctx, item)
//		require.NoError(t, err)
//	}
//
//	h := NewFilterHandler(sch)
//	h.SetQueryBuilder(NewMongoQueryBuilder())
//
//	t.Run("By Color [Equal/NotEqual]", func(t *testing.T) {
//		t.Run("Red", func(t *testing.T) {
//			query := h.Query(&Filter{Op: Equal, Field: "color", Value: "red"})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 2)
//			assert.ElementsMatch(t, []interface{}{"apple", "tomato"}, []interface{}{data[0]["name"], data[1]["name"]})
//		})
//		t.Run("Not Red", func(t *testing.T) {
//			query := h.Query(&Filter{Op: NotEqual, Field: "color", Value: "red"})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 1)
//			assert.Equal(t, "orange", data[0]["name"])
//		})
//	})
//	t.Run("By Quantity [Less/Greater]", func(t *testing.T) {
//		query := h.Query(&Filter{Op: LessOrEqual, Field: "qty", Value: 25}, &Filter{Op: Greater, Field: "qty", Value: 1})
//		res, err := coll.Find(ctx, query)
//		require.NoError(t, err)
//
//		var data []map[string]interface{}
//		err = res.All(ctx, &data)
//		require.NoError(t, err)
//		require.Len(t, data, 2)
//		assert.ElementsMatch(t, []interface{}{"apple", "orange"}, []interface{}{data[0]["name"], data[1]["name"]})
//	})
//	t.Run("Not Fruit [Equal embedded field]", func(t *testing.T) {
//		query := h.Query(&Filter{Op: Equal, Field: "info.is_fruit", Value: false})
//		res, err := coll.Find(ctx, query)
//		require.NoError(t, err)
//
//		var data []map[string]interface{}
//		err = res.All(ctx, &data)
//		require.NoError(t, err)
//		require.Len(t, data, 1)
//		assert.Equal(t, "tomato", data[0]["name"])
//	})
//	t.Run("By Similar [In/NotIn]", func(t *testing.T) {
//		t.Run("Similar to cucumber, pear", func(t *testing.T) {
//			query := h.Query(&Filter{Op: In, Field: "info.similar.name", Value: []string{"cucumber", "pear"}})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 2)
//			assert.ElementsMatch(t, []interface{}{"apple", "tomato"}, []interface{}{data[0]["name"], data[1]["name"]})
//		})
//		t.Run("Not Similar to cucumber, pear", func(t *testing.T) {
//			query := h.Query(&Filter{Op: NotIn, Field: "info.similar.name", Value: []string{"cucumber", "grapefruit"}})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 1)
//			assert.Equal(t, "apple", data[0]["name"])
//		})
//	})
//	t.Run("By Description [Contains/NotContains]", func(t *testing.T) {
//		t.Run("Contains", func(t *testing.T) {
//			query := h.Query(&Filter{Op: And, Value: []*Filter{
//				&Filter{Op: In, Field: "info.similar.color", Value: []string{"yellow"}},
//				&Filter{Op: Contains, Field: "info.desc", Value: "edible fruit"},
//			}})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 2)
//			assert.ElementsMatch(t, []interface{}{"apple", "orange"}, []interface{}{data[0]["name"], data[1]["name"]})
//		})
//		t.Run("Not Contains", func(t *testing.T) {
//			query := h.Query(&Filter{Op: NotContains, Field: "info.desc", Value: "fruit"})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			for _, d := range data {
//				fmt.Println(d["name"])
//			}
//			require.Len(t, data, 1)
//			assert.Equal(t, "tomato", data[0]["name"])
//		})
//	})
//	t.Run("By Shipment [Contains/NotContains]", func(t *testing.T) {
//		t.Run("Contains", func(t *testing.T) {
//			query := h.Query(
//				&Filter{Op: Contains, Field: "shipment", Value: "Russia"},
//			)
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			require.Len(t, data, 2)
//			assert.ElementsMatch(t, []interface{}{"apple", "tomato"}, []interface{}{data[0]["name"], data[1]["name"]})
//		})
//		t.Run("Not Contains", func(t *testing.T) {
//			query := h.Query(&Filter{Op: NotContains, Field: "shipment", Value: "Iran"})
//			res, err := coll.Find(ctx, query)
//			require.NoError(t, err)
//
//			var data []map[string]interface{}
//			err = res.All(ctx, &data)
//			require.NoError(t, err)
//			for _, d := range data {
//				fmt.Println(d["name"])
//			}
//			require.Len(t, data, 1)
//			assert.Equal(t, "tomato", data[0]["name"])
//		})
//	})
//	t.Run("Compound Query", func(t *testing.T) {
//		query := h.Query(&Filter{Op: Or, Value: []*Filter{
//			&Filter{Op: And, Value: []*Filter{
//				&Filter{Op: In, Field: "color", Value: []interface{}{"red", "yellow", "green"}},
//				&Filter{Op: Less, Field: "qty", Value: 10},
//			}}, // 1 - tomato
//			&Filter{Op: Equal, Field: "name", Value: "pepper"}, // 0
//			&Filter{Op: And, Value: []*Filter{
//				&Filter{Op: GreaterOrEqual, Field: "produced", Value: w1},
//				&Filter{Op: Less, Field: "produced", Value: w2}, // 1 - apple
//			}},
//		}})
//		res, err := coll.Find(ctx, query)
//		require.NoError(t, err)
//
//		var data []map[string]interface{}
//		err = res.All(ctx, &data)
//		require.NoError(t, err)
//		require.Len(t, data, 2)
//		assert.ElementsMatch(t, []interface{}{"apple", "tomato"}, []interface{}{data[0]["name"], data[1]["name"]})
//	})
//}
