package files

import (
	"fmt"
	"io/fs"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/id"
)

const (
	TemporaryPrefix = "tmp"
)

// File - описание файла в системе хранения perxis
type File struct {
	ID       string  `mapstructure:"id,omitempty" json:"id"`                                       // Уникальный идентификатор файла в хранилище
	Name     string  `mapstructure:"name,omitempty" json:"name" bson:"name,omitempty"`             // Имя файла
	Size     int     `mapstructure:"size,omitempty" json:"size" bson:"size,omitempty"`             // Размер файла
	MimeType string  `mapstructure:"mimeType,omitempty" json:"mimeType" bson:"mimeType,omitempty"` // Mime-type файла
	URL      string  `mapstructure:"url,omitempty" json:"url" bson:"url,omitempty"`                // Адрес для загрузки файла
	Key      string  `mapstructure:"key,omitempty" json:"key" bson:"key,omitempty"`                // Ключ для хранения файла в хранилище
	File     fs.File `mapstructure:"-" json:"-" bson:"-"`                                          // Файл для загрузки(из файловой системы)
}

func (f File) Clone() *File {
	return &f
}

func (f File) Temporary() bool {
	return strings.HasPrefix(f.ID, TemporaryPrefix)
}

func (f File) Fetch(i interface{}) interface{} {
	p, _ := i.(string)
	switch p {
	case "id":
		return f.ID
	case "name":
		return f.Name
	case "size":
		return f.Size
	case "mime_type":
		return f.MimeType
	case "url":
		return f.URL
	case "key":
		return f.Key
	default:
		panic("unknown parameter")
	}
}

func NewFile(name, mimeType string, size int, temp bool) *File {
	i := id.GenerateNewID()
	if temp {
		i = fmt.Sprintf("%s%s", TemporaryPrefix, i)
	}
	return &File{
		ID:       i,
		Name:     name,
		Size:     size,
		MimeType: mimeType,
	}
}

// MultipartUpload - описание загрузки файла
type MultipartUpload struct {
	File
	UploadID string           `json:"upload_id"` // Идентификатор загрузки хранилища
	PartSize int              `json:"part_size"` // Размер блока для загрузки
	PartURLs []string         `json:"part_urls"` // Адреса для загрузки полного файла
	Parts    []*CompletedPart `json:"parts"`     // Идентификаторы загруженных блоков (S3 ETAGs)
}

// Upload - описание загрузки файла
type Upload struct {
	File
	UploadURL string `json:"upload_url"` // Идентификатор загрузки хранилища
}

type CompletedPart struct {
	Number int    `json:"part_number"`
	ID     string `json:"id"`
}

func (u MultipartUpload) Clone() *MultipartUpload {
	return &u
}
