package extension

import (
	"context"
	"fmt"
	"reflect"
	"strings"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
)

func TestGetResults(t *testing.T) {

	getDummyExtension := func(name string, wantErr ...bool) Extension {
		ext := &testServerExtension{name: name}

		if len(wantErr) > 0 {
			ext.err = errors.WithDetail(errors.New("some err"), "Ошибка")
		}

		return ext
	}

	tests := []struct {
		name       string
		services   []Extension
		extensions []string
		fn         func(svc Extension) error
		want       []*RequestResult
	}{
		{
			name:       "one extension without errors",
			services:   []Extension{getDummyExtension("a"), getDummyExtension("b")},
			extensions: []string{"a"},
			fn:         func(svc Extension) error { return nil },
			want: []*RequestResult{
				{Extension: "a", State: RequestOK},
			},
		},
		{
			name:       "multiple extensions without errors",
			services:   []Extension{getDummyExtension("a"), getDummyExtension("b"), getDummyExtension("c")},
			extensions: []string{"a", "c"},
			fn:         func(svc Extension) error { return nil },
			want: []*RequestResult{
				{Extension: "a", State: RequestOK},
				{Extension: "c", State: RequestOK},
			},
		},
		{
			name:       "multiple extensions, one returns error",
			services:   []Extension{getDummyExtension("a"), getDummyExtension("b"), getDummyExtension("c", true)},
			extensions: []string{"a", "c"},
			fn:         func(svc Extension) error { return svc.Install(nil, nil) },
			want: []*RequestResult{
				{Extension: "a", State: RequestOK},
				{Extension: "c", State: RequestError, Error: "some err", Msg: "Ошибка\n"},
			},
		},
		{
			name:       "multiple extensions, all return error",
			services:   []Extension{getDummyExtension("a", true), getDummyExtension("b", true), getDummyExtension("c", true)},
			extensions: []string{"a", "b", "c"},
			fn:         func(svc Extension) error { return svc.Install(nil, nil) },
			want: []*RequestResult{
				{Extension: "a", State: RequestError, Error: "some err", Msg: "Ошибка\n"},
				{Extension: "b", State: RequestError, Error: "some err", Msg: "Ошибка\n"},
				{Extension: "c", State: RequestError, Error: "some err", Msg: "Ошибка\n"},
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			srv := NewServer(tt.services...)
			if got := srv.getResults(tt.extensions, tt.fn); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("getResults() = %v, want %v", got, tt.want)
			}
		})
	}
}

// не подходит использование mock.Extension из-за возникающих циклических импортов
type testServerExtension struct {
	err  error
	name string
}

func (t testServerExtension) GetDescriptor() *ExtensionDescriptor {
	return &ExtensionDescriptor{
		Extension:   t.name,
		Title:       strings.ToTitle(t.name),
		Description: "test extension",
		Version:     "0.0.0",
	}
}

func (t testServerExtension) Install(ctx context.Context, in *InstallRequest) error     { return t.err }
func (t testServerExtension) Check(ctx context.Context, in *CheckRequest) error         { return t.err }
func (t testServerExtension) Update(ctx context.Context, in *UpdateRequest) error       { return t.err }
func (t testServerExtension) Uninstall(ctx context.Context, in *UninstallRequest) error { return t.err }
func (t testServerExtension) Action(ctx context.Context, in *ActionRequest) (*ActionResponse, error) {
	return &ActionResponse{}, t.err
}

func TestServer_Action(t *testing.T) {
	getDummyExtension := func(name string, wantErr ...bool) Extension {
		ext := &testServerExtension{name: name}

		if len(wantErr) > 0 {
			ext.err = errors.WithDetail(errors.New("some err"), "Ошибка")
		}

		return ext
	}

	var tests = []struct {
		name     string
		services map[string]Extension
		in       *ActionRequest
		want     *ActionResponse
		wantErr  assert.ErrorAssertionFunc
	}{
		{
			name:     "OK (grpc)",
			services: map[string]Extension{"test-extension": getDummyExtension("test-extension")},
			in: &ActionRequest{
				Action:  "grpc:///test-extension/test-action",
				SpaceId: "sp",
				EnvId:   "env",
			},
			want:    &ActionResponse{State: ResponseDone},
			wantErr: assert.NoError,
		},
		{
			name:     "OK (deprecated call)",
			services: map[string]Extension{"test-extension": getDummyExtension("test-extension")},
			in: &ActionRequest{
				Action:    "test-action",
				SpaceId:   "sp",
				EnvId:     "env",
				Extension: "test-extension",
			},
			want:    &ActionResponse{State: ResponseDone},
			wantErr: assert.NoError,
		},
		{
			name:     "Error (unknown extension)",
			services: map[string]Extension{"test-extension": getDummyExtension("test-extension")},
			in: &ActionRequest{
				Action:  "grpc:///test-extension-2/test-action",
				SpaceId: "sp",
				EnvId:   "env",
			},
			want:    nil,
			wantErr: assert.Error,
		},
		{
			name:     "Error (deprecated call, no extension)",
			services: map[string]Extension{"test-extension": getDummyExtension("test-extension")},
			in: &ActionRequest{
				Action:  "test-action",
				SpaceId: "sp",
				EnvId:   "env",
			},
			want:    nil,
			wantErr: assert.Error,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {

			srv := &Server{
				services: tt.services,
			}
			got, err := srv.Action(context.Background(), tt.in)
			if !tt.wantErr(t, err, fmt.Sprintf("Action(%v)", tt.in)) {
				return
			}
			assert.Equalf(t, tt.want, got, "Action(%v)", tt.in)
		})
	}
}
