package extension

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/auth"
	pb "git.perx.ru/perxis/perxis-go/proto/extensions"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

type ManagerClient struct {
	*Client
	manager pb.ExtensionManagerClient
	conn    *grpc.ClientConn
}

func NewManagerClientWithAddr(addr string) (*ManagerClient, error) {
	cc, err := grpc.Dial(addr,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithUnaryInterceptor(auth.PrincipalClientInterceptor()),
	)
	if err != nil {
		return nil, err
	}

	return NewManagerClient(cc), nil
}

func NewManagerClient(cc *grpc.ClientConn) *ManagerClient {
	return &ManagerClient{
		manager: pb.NewExtensionManagerClient(cc),
		conn:    cc,
		Client:  NewClient(cc),
	}
}

func (c *ManagerClient) Close() error {
	return c.conn.Close()
}

func (c *ManagerClient) RegisterExtensions(ctx context.Context, exts ...*ExtensionConnector) error {
	var descs []*pb.ExtensionDescriptor
	for _, e := range exts {
		descs = append(descs, e.Descriptor)
	}
	_, err := c.manager.RegisterExtensions(ctx, &pb.RegisterExtensionsRequest{Extensions: descs}, grpc.WaitForReady(true))
	return err
}

func (c *ManagerClient) UnregisterExtensions(ctx context.Context, exts ...*ExtensionConnector) error {
	var descs []*pb.ExtensionDescriptor
	for _, e := range exts {
		descs = append(descs, e.Descriptor)
	}
	_, err := c.manager.UnregisterExtensions(ctx, &pb.UnregisterExtensionsRequest{Extensions: descs}, grpc.WaitForReady(true))
	return err
}

func (c *ManagerClient) ListExtensions(ctx context.Context, filter *ListExtensionsFilter) ([]*ExtensionConnector, error) {
	resp, err := c.manager.ListExtensions(ctx, &pb.ListExtensionsRequest{Filter: filter}, grpc.WaitForReady(true))
	if err != nil {
		return nil, err
	}

	var exts []*ExtensionConnector
	for _, desc := range resp.Extensions {
		exts = append(exts, &ExtensionConnector{Descriptor: desc})
	}

	return exts, nil
}
