package extension

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/auth"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	pb "git.perx.ru/perxis/perxis-go/proto/extensions"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
)

type Manager interface {
	Extension
	RegisterExtensions(ctx context.Context, ext ...*ExtensionConnector) error
	UnregisterExtensions(ctx context.Context, ext ...*ExtensionConnector) error
	ListExtensions(ctx context.Context, filter *ListExtensionsFilter) ([]*ExtensionConnector, error)
}

type (
	ListExtensionsFilter = pb.ListExtensionsFilter
)

type ExtensionConnector struct {
	Extension
	Connection *grpc.ClientConn
	Descriptor *ExtensionDescriptor
}

func (e *ExtensionConnector) Connect() error {
	// Уже есть экземпляр расширение или клиент расширения
	if e.Extension != nil {
		return nil
	}

	// Уже есть соединение, создаем клиента расширения
	if e.Connection != nil {
		if e.Extension == nil {
			e.Extension = NewClient(e.Connection)
			return nil
		}
		return nil
	}

	// Нет дескриптора или Url для соединения с расширением
	if e.Descriptor == nil || e.Descriptor.Url == "" {
		return nil
	}

	conn, err := grpc.Dial(
		e.Descriptor.Url,
		grpc.WithTransportCredentials(insecure.NewCredentials()),
		grpc.WithChainUnaryInterceptor(
			auth.PrincipalClientInterceptor(),
		),
	)
	if err != nil {
		return err
	}

	e.Connection = conn
	e.Extension = NewClient(conn)
	return nil
}

func (e *ExtensionConnector) IsConnected() bool {
	if e != nil && e.Connection != nil {
		return true
	}
	return false
}

func (e *ExtensionConnector) Close() error {
	if e.IsConnected() {
		if err := e.Connection.Close(); err != nil {
			return errors.Wrapf(err, "fail to close connection, extension %s (%s)", e.Extension, e.Descriptor.Url)
		}
		e.Connection = nil
	}
	return nil
}
