package extension

import (
	"crypto/sha512"
	"encoding/base64"
	"strings"
)

// KeyFunc функция получения ключа доступа к указанному пространству
type KeyFunc func(spaceID string) string
type SignatureFunc func(spaceID string) string

func NamedSignedKey(name string, signature any) KeyFunc {
	switch v := signature.(type) {
	case string:
		return func(spaceID string) string {
			return Sign(name, spaceID, v)
		}
	case SignatureFunc:
		return func(spaceID string) string {
			return Sign(name, spaceID, v(spaceID))
		}
	}
	panic("incorrect signature")
}

func ExtensionSignedKey(ext, signature any) KeyFunc {
	var desc *ExtensionDescriptor

	switch v := ext.(type) {
	case Extension:
		desc = v.GetDescriptor()
	case *ExtensionDescriptor:
		desc = v
	}

	switch v := signature.(type) {
	case string:
		return func(spaceID string) string {
			return Sign(desc.Extension, spaceID, v)
		}
	case SignatureFunc:
		return func(spaceID string) string {
			return Sign(desc.Extension, spaceID, v(spaceID))
		}
	}
	panic("incorrect signature")
}

func Sign(vals ...string) string {
	value := strings.Join(vals, ".")
	hash := sha512.Sum512([]byte(value))
	encoded := base64.StdEncoding.EncodeToString(hash[:])
	return encoded
}
