package expr

import (
	"time"

	"github.com/antonmedv/expr"
)

const DefaultTimeLayout = time.RFC3339
const DefaultDateLayout = "2006-01-02"

type TimeExpr struct{}

func init() {
	RegisterOption(
		Extend("Time", TimeExpr{}),
		expr.Operator("==", "Time.Equal"),
		expr.Operator("<", "Time.Before"),
		expr.Operator(">", "Time.After"),
		expr.Operator("<=", "Time.BeforeOrEqual"),
		expr.Operator(">", "Time.After"),
		expr.Operator(">=", "Time.AfterOrEqual"),

		// Time and duration manipulation.
		expr.Operator("+", "Time.Add"),
		expr.Operator("-", "Time.Sub"),

		// Operators override for duration comprising.
		expr.Operator("==", "Time.EqualDuration"),
		expr.Operator("<", "Time.BeforeDuration"),
		expr.Operator("<=", "Time.BeforeOrEqualDuration"),
		expr.Operator(">", "Time.AfterDuration"),
		expr.Operator(">=", "Time.AfterOrEqualDuration"),
	)
}

func (TimeExpr) Time(v interface{}) time.Time {
	switch val := v.(type) {

	case string:
		if t, err := time.Parse(DefaultTimeLayout, val); err != nil {
			panic(err)
		} else {
			return t
		}
	case time.Time:
		return v.(time.Time)
	}
	return v.(time.Time)
}

func (TimeExpr) Date(s string) time.Time {
	t, err := time.Parse(DefaultDateLayout, s)
	if err != nil {
		panic(err)
	}
	return t
}
func (TimeExpr) Duration(s string) time.Duration {
	d, err := time.ParseDuration(s)
	if err != nil {
		panic(err)
	}
	return d
}

func (TimeExpr) Now() time.Time                                { return time.Now() }
func (TimeExpr) Equal(a, b time.Time) bool                     { return a.Equal(b) }
func (TimeExpr) Before(a, b time.Time) bool                    { return a.Before(b) }
func (TimeExpr) BeforeOrEqual(a, b time.Time) bool             { return a.Before(b) || a.Equal(b) }
func (TimeExpr) After(a, b time.Time) bool                     { return a.After(b) }
func (TimeExpr) AfterOrEqual(a, b time.Time) bool              { return a.After(b) || a.Equal(b) }
func (TimeExpr) Add(a time.Time, b time.Duration) time.Time    { return a.Add(b) }
func (TimeExpr) Sub(a, b time.Time) time.Duration              { return a.Sub(b) }
func (TimeExpr) EqualDuration(a, b time.Duration) bool         { return a == b }
func (TimeExpr) BeforeDuration(a, b time.Duration) bool        { return a < b }
func (TimeExpr) BeforeOrEqualDuration(a, b time.Duration) bool { return a <= b }
func (TimeExpr) AfterDuration(a, b time.Duration) bool         { return a > b }
func (TimeExpr) AfterOrEqualDuration(a, b time.Duration) bool  { return a >= b }
