package expr

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"golang.org/x/net/context"
)

func TestSlice(t *testing.T) {
	ctx := context.Background()

	tests := []struct {
		name  string
		eval  string
		env   map[string]interface{}
		wantB interface{}
	}{
		// nil
		{"nil#1", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": nil}, false},
		{"nil#2", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": []interface{}{"a", "b"}}, false},
		{"nil#3", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": []interface{}{1, 2}}, false},
		{"nil#4", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": []interface{}{1.1, 1.2}}, false},
		{"nil#5", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": "a"}, false},
		{"nil#6", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": 1}, false},
		{"nil#7", "In(a1, a2)", map[string]interface{}{"a1": nil, "a2": 1.1}, false},
		// strings
		{"strings#1", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"a", "b"}, "a2": []interface{}{"a", "b"}}, true},
		{"strings#2", "In(a1, a2)", map[string]interface{}{"a1": "b", "a2": []interface{}{"a", "b"}}, true},
		{"strings#3", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"a", "b"}, "a2": "b"}, true},
		{"strings#4", "In(a1, a2)", map[string]interface{}{"a1": "b", "a2": "b"}, true},
		{"strings#5", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"a", "b"}, "a2": nil}, false},
		{"strings#6", "In(a1, a2)", map[string]interface{}{"a1": "c", "a2": []interface{}{"a", "b"}}, false},
		{"strings#7", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"a", "b"}, "a2": "c"}, false},
		{"strings#8", "In(a1, a2)", map[string]interface{}{"a1": "a", "a2": "b"}, false},
		{"strings#9", "In(a1, a2)", map[string]interface{}{"a1": "1", "a2": 1}, false},
		// numbers
		{"numbers#1", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{1, 2}, "a2": []interface{}{1, 2}}, true},
		{"numbers#5", "In(a1, a2)", map[string]interface{}{"a1": 1, "a2": []interface{}{1, 2}}, true},
		{"numbers#6", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{1, 2}, "a2": 2}, true},
		{"numbers#7", "In(a1, a2)", map[string]interface{}{"a1": 3, "a2": []interface{}{1, 2}}, false},
		{"numbers#8", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{1, 2}, "a2": 3}, false},
		{"numbers#9", "In(a1, a1)", map[string]interface{}{"a1": []interface{}{1.1, 2.1}, "a2": []interface{}{1.1, 2.1}}, true},
		{"numbers#10", "In(a1, a2)", map[string]interface{}{"a1": 1.1, "a2": []interface{}{1.1, 2.1}}, true},
		{"numbers#11", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{1.1, 2.1}, "a2": 2.1}, true},
		{"numbers#12", "In(a1, a2)", map[string]interface{}{"a1": 3.1, "a2": []interface{}{1.1, 2.1}}, false},
		{"numbers#13", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{1.1, 2.1}, "a2": 3.1}, false},
		// mix types
		{"mix#1", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"a", "b"}, "a2": map[string]interface{}{"a": "a", "b": "b"}}, false},
		{"mix#2", "In(a1, a2)", map[string]interface{}{"a1": 2, "a2": "2"}, false},
		{"mix#3", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{}, "a2": nil}, false},
		{"mix#4", "In(a1, a2)", map[string]interface{}{"a1": []interface{}{"1", "2"}, "a2": []interface{}{1, 2}}, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotB, err := Eval(ctx, tt.eval, tt.env)
			require.NoError(t, err)
			assert.Equal(t, tt.wantB, gotB)
		})
	}
}
