package expr

import (
	"fmt"
	"path"
	"regexp"
	"strings"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/id"
	"github.com/gosimple/slug"
)

var markersRE = regexp.MustCompile(`{\w+}`)

func init() {
	RegisterOption(
		Extend(
			"sprintf", Sprintf,
			"trim_space", TrimSpace,
			"to_upper", ToUpper,
			"slugify", Slugify,
			"make_path", MakePath,
			"normalize_string", func(s string) string {
				s = strings.TrimSpace(strings.ToLower(s))
				s = strings.ReplaceAll(s, " ", "_")
				s = strings.ReplaceAll(s, ".", "_")
				s = strings.ReplaceAll(s, "/", "_")
				return strings.ToValidUTF8(s, "")
			},
			"replace_markers", ReplaceMarkers,
		),
	)
}

func Slugify(value interface{}) string {
	if s, ok := value.(string); ok {
		return slug.Make(s)
	}
	return ""
}

func MakePath(base, url interface{}) string {
	b, p := "/", ""
	if s, ok := base.(string); ok && s != "" {
		b = s
	}

	if s, ok := url.(string); ok {
		p = s
	}

	return path.Join(b, p)
}

func ToUpper(value interface{}) string {
	if s, ok := value.(string); ok {
		return strings.ToUpper(s)
	}
	return ""
}

func TrimSpace(value interface{}) string {
	if s, ok := value.(string); ok {
		return strings.TrimSpace(s)
	}
	return ""
}

func Sprintf(format interface{}, a ...interface{}) string {
	if f, ok := format.(string); ok {
		return fmt.Sprintf(f, a...)
	}
	return ""
}

func ReplaceMarkers(value interface{}) string {
	if s, ok := value.(string); ok {
		s = markersRE.ReplaceAllStringFunc(s, func(m string) string {
			switch {
			case m == "{new_id}":
				return id.GenerateNewID()
			case m == "{now}":
				return time.Now().Format(time.RFC3339)
			}
			return m
		})
		return s
	}
	return ""
}
