package events

import (
	"reflect"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/nats-io/nats.go"
)

type natsConnetion struct {
	Conn *nats.Conn
	enc  nats.Encoder
	// добавление префикса для всех топиков
	prefix string
}

func Open(url string, prefix string) (Connection, error) {
	var err error
	b := new(natsConnetion)
	b.Conn, err = nats.Connect(url)
	if err != nil {
		return nil, err
	}
	b.enc = &ProtobufEncoder{}
	b.prefix = prefix
	return b, nil
}

func (c *natsConnetion) getSubject(subject string) string {
	if c.prefix != "" {
		subject = c.prefix + "." + subject
	}
	return subject
}

func (c *natsConnetion) Publish(subject string, msg any, opts ...PublishOption) error {
	m := &nats.Msg{Subject: c.getSubject(subject)}
	switch v := msg.(type) {
	case *nats.Msg:
		m = v
	case []byte:
		m.Data = v
	default:
		data, err := c.enc.Encode(subject, v)
		if err != nil {
			return err
		}
		m.Data = data
	}

	filters := PublishFilters(NewPublishOptions(opts...))
	if len(filters) > 0 {
		for _, f := range filters {
			if m = f(m); m == nil {
				return nil
			}
		}
	}

	return c.Conn.PublishMsg(m)
}

func (c *natsConnetion) Subscribe(subject string, handler any, opts ...SubscribeOption) (Subscription, error) {

	subject = c.getSubject(subject)
	return c.subscribe(subject, handler, SubscribeFilters(NewSubscribeOptions(opts...)))
}

func (c *natsConnetion) Close() (err error) {
	if err = c.Conn.Drain(); err != nil {
		return err
	}
	c.Conn.Close()
	return
}

// Dissect the cb Handler's signature
func argInfo(cb nats.Handler) (reflect.Type, int) {
	cbType := reflect.TypeOf(cb)
	if cbType.Kind() != reflect.Func {
		panic("handler needs to be a func")
	}
	numArgs := cbType.NumIn()
	if numArgs == 0 {
		return nil, numArgs
	}
	return cbType.In(numArgs - 1), numArgs
}

var emptyMsgType = reflect.TypeOf(&nats.Msg{})

type MsgFilter func(*nats.Msg) *nats.Msg

// Internal implementation that all public functions will use.
func (c *natsConnetion) subscribe(subject string, cb nats.Handler, filters []MsgFilter) (*nats.Subscription, error) {
	if cb == nil {
		return nil, errors.New("handler required for subscription")
	}
	argType, numArgs := argInfo(cb)
	if argType == nil {
		return nil, errors.New("handler requires at least one argument")
	}

	cbValue := reflect.ValueOf(cb)
	wantsRaw := (argType == emptyMsgType)

	natsCB := func(m *nats.Msg) {
		if len(filters) > 0 {
			for _, f := range filters {
				if m = f(m); m == nil {
					return
				}
			}
		}

		var oV []reflect.Value
		if wantsRaw {
			oV = []reflect.Value{reflect.ValueOf(m)}
		} else {
			var oPtr reflect.Value
			if argType.Kind() != reflect.Ptr {
				oPtr = reflect.New(argType)
			} else {
				oPtr = reflect.New(argType.Elem())
			}
			if err := c.enc.Decode(m.Subject, m.Data, oPtr.Interface()); err != nil {
				if errorHandler := c.Conn.ErrorHandler(); errorHandler != nil {
					errorHandler(c.Conn, m.Sub, errors.Wrap(err, "Got an unmarshal error"))
				}
				return
			}
			if argType.Kind() != reflect.Ptr {
				oPtr = reflect.Indirect(oPtr)
			}

			switch numArgs {
			case 1:
				oV = []reflect.Value{oPtr}
			case 2:
				subV := reflect.ValueOf(m.Subject)
				oV = []reflect.Value{subV, oPtr}
			case 3:
				subV := reflect.ValueOf(m.Subject)
				replyV := reflect.ValueOf(m.Reply)
				oV = []reflect.Value{subV, replyV, oPtr}
			}

		}
		cbValue.Call(oV)
	}

	return c.Conn.Subscribe(subject, natsCB)
}

func PublishFilters(opts *PublishOptions) []MsgFilter {
	if opts == nil {
		return nil
	}
	var filters []MsgFilter

	if len(opts.Tags) > 0 {
		filters = append(filters, func(msg *nats.Msg) *nats.Msg {
			if msg.Header == nil {
				msg.Header = make(nats.Header)
			}
			for _, v := range opts.Tags {
				msg.Header.Add("Tag", v)
			}
			return msg
		})
	}

	return filters
}

func SubscribeFilters(opts *SubscribeOptions) []MsgFilter {
	if opts == nil {
		return nil
	}
	var filters []MsgFilter

	if len(opts.FilterTags) > 0 {
		filters = append(filters, func(msg *nats.Msg) *nats.Msg {
			tags := msg.Header.Values("Tag")
			for _, tag := range tags {
				for _, v := range opts.FilterTags {
					if v == tag {
						return msg
					}
				}
			}
			return nil
		})
	}

	return filters
}
