package errors

import (
	"errors"
	"fmt"
	"io"

	"google.golang.org/grpc/codes"
)

type ErrStatus interface {
	StatusCode() codes.Code
	Error() string
}

type withStatusError struct {
	err  error
	code codes.Code
}

func (w *withStatusError) StatusCode() codes.Code { return w.code }
func (w *withStatusError) Error() string          { return w.err.Error() }
func (w *withStatusError) Unwrap() error          { return w.err }

func (w *withStatusError) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Unwrap())
			fmt.Fprintf(s, "\nerror status: %s", w.code)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithStatusCode(err error, code codes.Code) error {
	if err == nil || code == codes.Unknown {
		return err
	}

	var se *withStatusError
	if errors.As(err, &se) {
		if se.StatusCode() == code {
			return err
		}
	}

	return &withStatusError{
		err:  err,
		code: code,
	}
}

func GetStatusCode(err error) codes.Code {
	for e := err; e != nil; e = Unwrap(e) {
		if errStatus, ok := e.(ErrStatus); ok {
			return errStatus.StatusCode()
		}
	}
	return codes.Unknown
}

func InvalidArgument(err error) error    { return WithStatusCode(err, codes.InvalidArgument) }
func DeadlineExceeded(err error) error   { return WithStatusCode(err, codes.DeadlineExceeded) }
func NotFound(err error) error           { return WithStatusCode(err, codes.NotFound) }
func AlreadyExists(err error) error      { return WithStatusCode(err, codes.AlreadyExists) }
func PermissionDenied(err error) error   { return WithStatusCode(err, codes.PermissionDenied) }
func FailedPrecondition(err error) error { return WithStatusCode(err, codes.FailedPrecondition) }
func Internal(err error) error           { return WithStatusCode(err, codes.Internal) }
func Unavailable(err error) error        { return WithStatusCode(err, codes.Unavailable) }
func Unauthenticated(err error) error    { return WithStatusCode(err, codes.Unauthenticated) }
