package errors

import (
	"fmt"
	"io"
)

type MultiError interface {
	Errors() []error
	Error() string
}

type withMultiError struct {
	err    error
	errors []error
}

func (w *withMultiError) Errors() []error { return w.errors }
func (w *withMultiError) Error() string   { return w.err.Error() }
func (w *withMultiError) Unwrap() error   { return w.err }

func (w *withMultiError) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Unwrap())
			if len(w.errors) > 0 {
				fmt.Fprint(s, "\ninclude errors:\n")
				for _, e := range w.errors {
					fmt.Fprintf(s, "  %s\n", e)
				}
			}

			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithErrors(err error, errs ...error) error {
	if err == nil || len(errs) == 0 {
		return err
	}
	return &withMultiError{err: err, errors: errs}
}

func GetErrors(err error) []error {
	for e := err; e != nil; e = Unwrap(e) {
		if errMulti, ok := e.(MultiError); ok {
			return errMulti.Errors()
		}
	}
	return nil
}
