package grpc

import (
	"context"
	"testing"

	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/proto/common"
	"github.com/go-kit/kit/endpoint"
	"github.com/hashicorp/go-multierror"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"
)

func testEndopoint(e error) endpoint.Endpoint {
	return func(ctx context.Context, request interface{}) (response interface{}, err error) {
		return nil, e
	}
}

func TestMiddleware(t *testing.T) {
	merr := multierror.Append(errors.WithField(errors.New("err1"), "fld1"), errors.WithField(errors.New("err2"), "fld2"))
	err := errors.WrapErr(errors.New("test"), merr)
	err = errors.WithID(err)
	err = errors.NotFound(err)
	err = errors.Wrap(err, "wrapped")
	errID := errors.GetID(err)

	serverEp := ServerMiddleware(testEndopoint(err))
	_, grpcErr := serverEp(context.Background(), nil)
	st, _ := status.FromError(grpcErr)
	assert.Equal(t, "wrapped: test", st.Message())
	assert.Equal(t, codes.NotFound, st.Code())
	dt := st.Details()
	require.Len(t, dt, 1)
	require.IsType(t, (*common.Error)(nil), dt[0])

	clientEp := ClientMiddleware(testEndopoint(grpcErr))
	_, cliErr := clientEp(context.Background(), nil)
	assert.Equal(t, "wrapped: test", cliErr.Error(), "ожидается то же строковое значние, что и у изначальной ошибки")
	assert.Equal(t, errID, errors.GetID(cliErr))
	assert.Equal(t, codes.NotFound, errors.GetStatusCode(cliErr))
	assert.ErrorAs(t, err, &merr)
	assert.Len(t, merr.Errors, 2)
}
