package errors

import (
	"fmt"
	"io"

	"github.com/hashicorp/go-multierror"
)

type FieldError interface {
	Field() string
	Error() string
}

type withField struct {
	err   error
	field string
}

func (w *withField) Field() string { return w.field }
func (w *withField) Error() string { return fmt.Sprintf("field '%s': %s", w.field, w.err.Error()) }
func (w *withField) Unwrap() error { return w.err }

func (w *withField) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Unwrap())
			fmt.Fprintf(s, "\nfield: %s", w.field)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithField(err error, field string) error {
	if err == nil {
		return nil
	}

	var merr *multierror.Error
	if As(err, &merr) {
		for i, e := range merr.Errors {
			merr.Errors[i] = WithField(e, field)
		}
		return err
	}

	var ferr *withField
	if As(err, &ferr) {
		if ferr.field != "" {
			ferr.field = field + "." + ferr.field
		} else {
			ferr.field = field
		}
		return err
	}

	return &withField{
		field: field,
		err:   err,
	}
}

func GetField(err error) string {
	for e := err; e != nil; e = Unwrap(e) {
		if errField, ok := e.(FieldError); ok {
			return errField.Field()
		}
	}
	return ""
}
