package errors

import (
	"fmt"
	"io"
)

const (
	NoDomain     = ""
	GlobalDomain = "global"
)

type ErrDomain interface{ Domain() string }

type withDomain struct {
	err    error
	domain string
}

func (w *withDomain) Domain() string { return w.domain }
func (w *withDomain) Error() string  { return w.err.Error() }
func (w *withDomain) Unwrap() error  { return w.err }

func (w *withDomain) Format(s fmt.State, verb rune) {
	switch verb {
	case 'v':
		if s.Flag('+') {
			fmt.Fprintf(s, "%+v", w.Unwrap())
			fmt.Fprintf(s, "\nerror domain: %s", w.domain)
			return
		}
		fallthrough
	case 's':
		io.WriteString(s, w.Error())
	case 'q':
		fmt.Fprintf(s, "%q", w.Error())
	}
}

func WithDomain(err error, domain string) error {
	if err == nil {
		return nil
	}
	return &withDomain{err: err, domain: domain}
}

func Global(err error) error {
	return WithDomain(err, GlobalDomain)
}

func GetDomain(err error) string {
	for e := err; e != nil; e = Unwrap(e) {
		if errDomain, ok := e.(ErrDomain); ok {
			return errDomain.Domain()
		}
	}
	return NoDomain
}
