package environments

import "context"

type EnvironmentObserver interface{}

// EnvironmentCreatedObserver интерфейс наблюдателя вызываемый при создании окружения.
// Инициировать оповещение наблюдателя может вызов метода `Environments.Create`
// Оповещение выполнятся в асинхронном режиме.
type EnvironmentCreatedObserver interface {
	OnEnvironmentCreated(ctx context.Context, env *Environment) (delayedTaskID string, err error)
}

// EnvironmentUpdatedObserver интерфейс наблюдателя вызываемый при изменении окружения.
// Инициировать оповещение наблюдателя может вызов методов `Environments.Update`,
// `Environments.SetAlias`, `Environments.RemoveAlias`
// Оповещение выполнятся в асинхронном режиме.
type EnvironmentUpdatedObserver interface {
	OnEnvironmentUpdated(ctx context.Context, before, after *Environment) error
}

// EnvironmentDeletedObserver интерфейс наблюдателя вызываемый при удалении окружения.
// Инициировать оповещение наблюдателя может вызов метода `Environments.Delete`.
// Оповещение выполнятся в асинхронном режиме.
type EnvironmentDeletedObserver interface {
	OnEnvironmentDeleted(ctx context.Context, env *Environment) error
}

// EnvironmentConfigurationObserver интерфейс наблюдателя вызываемый при изменении конфигурации
// окружения.
// Инициировать оповещение наблюдателя может вызов методов `Environments.OnSpaceUpdate`,
// Оповещение выполнятся в асинхронном режиме.
type EnvironmentConfigurationObserver interface {
	OnEnvironmentConfiguration(ctx context.Context, before, after *Environment) error
}

// EnvironmentMigratedObserver интерфейс наблюдателя вызываемый при запуске миграции окружения.
// Инициировать оповещение наблюдателя может вызов методов `Environments.Migrate`,
type EnvironmentMigratedObserver interface {
	OnEnvironmentMigrated(ctx context.Context, env *Environment) error
}
