package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/environments"
	mocksenvironments "git.perx.ru/perxis/perxis-go/pkg/environments/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestEnvironmentsCache(t *testing.T) {

	const (
		envID    = "envID"
		spaceID  = "spaceID"
		envAlias = "envAlias"
		size     = 5
		ttl      = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		envs := &mocksenvironments.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

		envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}, nil).Once()

		v1, err := svc.Get(ctx, spaceID, envID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID, envID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша при повторном запросе по ID.")

		v3, err := svc.Get(ctx, spaceID, envAlias)
		require.NoError(t, err)
		assert.Same(t, v3, v2, "Ожидается получение объекта из кеша, при запросе того же объекта по alias окружения.")

		envs.AssertExpectations(t)
	})

	t.Run("Get from cache(by Alias)", func(t *testing.T) {
		envs := &mocksenvironments.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

		envs.On("Get", mock.Anything, spaceID, envAlias).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}, nil).Once()

		v1, err := svc.Get(ctx, spaceID, envAlias)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID, envAlias)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кэша по alias.")

		v3, err := svc.Get(ctx, spaceID, envID)
		require.NoError(t, err)
		assert.Same(t, v3, v2, "Ожидается получение объекта из кеша, при запросе того же объекта по ID окружения.")

		envs.AssertExpectations(t)
	})

	t.Run("List from cache", func(t *testing.T) {
		envs := &mocksenvironments.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

		envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment"}}, nil).Once()

		vl1, err := svc.List(ctx, spaceID)
		require.NoError(t, err)

		vl2, err := svc.List(ctx, spaceID)
		require.NoError(t, err)
		assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

		envs.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After SetAlias", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2"}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2"}}}, nil).Once()
			envs.On("SetAlias", mock.Anything, spaceID, envID, envAlias).Return(nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			err = svc.SetAlias(ctx, spaceID, envID, envAlias)
			require.NoError(t, err)

			envs.On("Get", mock.Anything, spaceID, envAlias).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2", envAlias}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2", envAlias}}}, nil).Once()

			v4, err := svc.Get(ctx, spaceID, envAlias)
			require.NoError(t, err)
			assert.Contains(t, v4.Aliases, envAlias, "Ожидает что элемент будет запрошен из сервиса по Alias.")

			v5, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v4, v5, "Ожидается получение объекта из кэша по ID.")

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидает что объекты будут удалены из кэша и запрошены из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After RemoveAlias", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2", envAlias}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2", envAlias}}}, nil).Once()
			envs.On("RemoveAlias", mock.Anything, spaceID, envID, envAlias).Return(nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша по ID.")

			v3, err := svc.Get(ctx, spaceID, envAlias)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по Alias.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			err = svc.RemoveAlias(ctx, spaceID, envID, envAlias)
			require.NoError(t, err)

			envs.On("Get", mock.Anything, spaceID, envAlias).Return(nil, errNotFound).Once()
			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2"}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{"envID2"}}}, nil).Once()

			_, err = svc.Get(ctx, spaceID, envAlias)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидает что элемент был удален из кеша и сервис вернул ошибку на запрос по несуществующему Alias.")

			v4, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.NotSame(t, v3, v4, "Ожидает что элемент был удален из кеша и получен из сервиса по ID.")

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидает что объекты будут удалены из кэша и запрошены из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After Update", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}}, nil).Once()
			envs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			v3, err := svc.Get(ctx, spaceID, envAlias)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по Alias.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			err = svc.Update(ctx, &environments.Environment{ID: envID, SpaceID: spaceID, Description: "EnvironmentUPD", Aliases: []string{envAlias}})
			require.NoError(t, err)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "EnvironmentUPD", Aliases: []string{envAlias}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "EnvironmentUPD", Aliases: []string{envAlias}}}, nil).Once()

			_, err = svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v4, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v4, "Ожидает что элемент был удален из кэша и будет запрошен заново из сервиса.")

			v5, err := svc.Get(ctx, spaceID, envAlias)
			require.NoError(t, err)
			assert.Same(t, v4, v5, "Ожидается получение объекта из кэша по Alias после обновления объекта и получения по ID.")

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидает что объекты будут удалены из кэша и запрошены из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After Update(List)", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}}, nil).Once()
			envs.On("Update", mock.Anything, mock.Anything).Return(nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			err = svc.Update(ctx, &environments.Environment{ID: envID, SpaceID: spaceID, Description: "EnvironmentUPD", Aliases: []string{envAlias}})
			require.NoError(t, err)

			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "EnvironmentUPD", Aliases: []string{envAlias}}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.NotSame(t, vl2[0], vl3[0], "Ожидает что объекты будут удалены из кэша и запрошены из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}, nil).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment", Aliases: []string{envAlias}}}, nil).Once()
			envs.On("Delete", mock.Anything, spaceID, envID).Return(nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			v3, err := svc.Get(ctx, spaceID, envAlias)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кэша по Alias.")

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			err = svc.Delete(ctx, spaceID, envID)
			require.NoError(t, err)

			envs.On("Get", mock.Anything, spaceID, envID).Return(nil, errNotFound).Once()
			envs.On("Get", mock.Anything, spaceID, envAlias).Return(nil, errNotFound).Once()
			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{}, nil).Once()

			_, err = svc.Get(ctx, spaceID, envID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидает что элемент был удален из кэша по ID и получена ошибка от сервиса.")

			_, err = svc.Get(ctx, spaceID, envAlias)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидает что элемент был удален  из кэша по Alias и получена ошибка от сервиса.")

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl3, 0, "Ожидает что объекты будут удалены из кэша и запрошены из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After Create", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment"}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кэша.")

			envs.On("Create", mock.Anything, mock.Anything).Return(&environments.Environment{ID: "envID2", SpaceID: spaceID, Description: "Environment2"}, nil).Once()
			_, err = svc.Create(ctx, &environments.Environment{ID: "envID2", SpaceID: spaceID, Description: "Environment2"})
			require.NoError(t, err)

			envs.On("List", mock.Anything, spaceID).Return([]*environments.Environment{{ID: envID, SpaceID: spaceID, Description: "Environment"}, {ID: "envID2", SpaceID: spaceID, Description: "Environment2"}}, nil).Once()

			vl3, err := svc.List(ctx, spaceID)
			require.NoError(t, err)
			assert.Len(t, vl3, 2, "Ожидает что объекты были удалены из кэша и запрошены заново из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After size exceeded", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(1, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, "envID2").Return(&environments.Environment{ID: "envID2", SpaceID: spaceID, Description: "Environment2"}, nil).Once()
			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment"}, nil).Once()
			envs.On("Get", mock.Anything, spaceID, "envID2").Return(&environments.Environment{ID: "envID2", SpaceID: spaceID, Description: "Environment2"}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, "envID2")
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, "envID2")
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			_, err = svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v5, err := svc.Get(ctx, spaceID, "envID2")
			require.NoError(t, err)
			assert.NotSame(t, v2, v5, "Ожидает что объект был удален из кэша и будет запрошен заново из сервиса.")

			envs.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			envs := &mocksenvironments.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl))(envs)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment"}, nil).Once()
			v1, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кэша.")

			time.Sleep(2 * ttl)

			envs.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Description: "Environment"}, nil).Once()
			v3, err := svc.Get(ctx, spaceID, envID)
			require.NoError(t, err)
			assert.NotSame(t, v2, v3, "Ожидает что объект был удален из кэша и будет запрошен заново из сервиса.")

			envs.AssertExpectations(t)
		})
	})
}
