package middleware

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/environments"
)

func makeKey(ss ...string) string {
	return strings.Join(ss, "-")
}

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Environments) service.Environments {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Environments
}

func (m cachingMiddleware) Create(ctx context.Context, env *service.Environment) (environment *service.Environment, err error) {

	environment, err = m.next.Create(ctx, env)
	if err == nil {
		m.cache.Remove(environment.SpaceID)
	}
	return environment, err
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId string, envId string) (environment *service.Environment, err error) {

	value, e := m.cache.Get(makeKey(spaceId, envId))
	if e == nil {
		return value.(*service.Environment), err
	}
	environment, err = m.next.Get(ctx, spaceId, envId)
	if err == nil {
		m.cache.Set(makeKey(spaceId, environment.ID), environment)
		for _, a := range environment.Aliases {
			m.cache.Set(makeKey(spaceId, a), environment)
		}
	}
	return environment, err
}

func (m cachingMiddleware) List(ctx context.Context, spaceId string) (environments []*service.Environment, err error) {

	value, e := m.cache.Get(spaceId)
	if e == nil {
		return value.([]*service.Environment), err
	}
	environments, err = m.next.List(ctx, spaceId)
	if err == nil {
		m.cache.Set(spaceId, environments)
	}
	return environments, err
}

func (m cachingMiddleware) Update(ctx context.Context, env *service.Environment) (err error) {

	err = m.next.Update(ctx, env)
	if err == nil {
		value, e := m.cache.Get(makeKey(env.SpaceID, env.ID))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}
		m.cache.Remove(env.SpaceID)
	}
	return err
}

func (m cachingMiddleware) Delete(ctx context.Context, spaceId string, envId string) (err error) {

	err = m.next.Delete(ctx, spaceId, envId)
	if err == nil {
		value, e := m.cache.Get(makeKey(spaceId, envId))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func (m cachingMiddleware) SetAlias(ctx context.Context, spaceId string, envId string, alias string) (err error) {

	err = m.next.SetAlias(ctx, spaceId, envId, alias)
	if err == nil {
		value, e := m.cache.Get(makeKey(spaceId, alias))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}

		value, e = m.cache.Get(makeKey(spaceId, envId))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func (m cachingMiddleware) RemoveAlias(ctx context.Context, spaceId string, envId string, alias string) (err error) {

	err = m.next.RemoveAlias(ctx, spaceId, envId, alias)
	if err == nil {
		m.cache.Remove(spaceId)
		value, e := m.cache.Get(makeKey(spaceId, alias))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}

		value, e = m.cache.Get(makeKey(spaceId, envId))
		if e == nil {
			env := value.(*service.Environment)
			m.cache.Remove(makeKey(env.SpaceID, env.ID))
			for _, a := range env.Aliases {
				m.cache.Remove(makeKey(env.SpaceID, a))
			}
		}
	}
	return err
}

func (m cachingMiddleware) Migrate(ctx context.Context, spaceId, envId string, options ...*service.MigrateOptions) (err error) {
	err = m.next.Migrate(ctx, spaceId, envId, options...)

	// значение из кэша удалить вне зависимости от наличия ошибки, поскольку состояние окружения могло измениться
	value, e := m.cache.Get(makeKey(spaceId, envId))
	if e == nil {
		env := value.(*service.Environment)
		m.cache.Remove(makeKey(env.SpaceID, env.ID))
		for _, a := range env.Aliases {
			m.cache.Remove(makeKey(env.SpaceID, a))
		}
	}
	m.cache.Remove(spaceId)
	return err
}
