package environments

import "time"

const (
	DefaultEnvironment = "master"
)

type State int

const (
	StateUnknown State = iota
	StateNew
	StatePreparing
	StateReady
	StateError

	StateInfoEmpty = "EMPTY"
)

func (s State) String() string {
	switch s {
	case StateNew:
		return "new"
	case StatePreparing:
		return "preparing"
	case StateReady:
		return "ready"
	case StateError:
		return "error"
	default:
		return "unknown"
	}
}

type StateInfo struct {
	State     State     `json:"state" bson:"state"`
	StartedAt time.Time `json:"started_at,omitempty" bson:"started_at,omitempty"`
	Info      string    `json:"info,omitempty" bson:"info,omitempty"`
}

type Config struct {
	SourceID string

	// Deprecated
	Features []string
}

// Environment - представляет рабочее окружения для пространства
// Каждое окружение может иметь собственный набор коллекций и данных и
// использоваться независимо друг от друга
type Environment struct {
	ID          string `json:"id" bson:"_id"` // Идентификатор окружения, задается пользователем при создании. Уникален в рамках пространства `SpaceID`
	SpaceID     string `json:"spaceID" bson:"-"`
	Description string `json:"description" bson:"desc,omitempty"` // Описание для окружения
	//State       State  `json:"state" bson:"state"`                // Состояние окружения (Preparing/Ready/Failed)
	//StateInfo   string   `json:"state_info,omitempty" bson:"state_info,omitempty"`

	// StateInfo отображает состояние коллекции:
	// - State: идентификатор состояния окружения (unknown/new/preparing/ready/error)
	// - Info: дополнительная информация о состоянии коллекции (например, если при
	//   применении схемы к коллекции произошла ошибка)
	// - StartedAt: время, в которое коллекция перешла в состояние `Preparing`
	StateInfo *StateInfo `json:"state_info" bson:"state_info,omitempty"`

	Aliases []string `json:"aliases" bson:"aliases,omitempty"` // Синонимы окружения (только чтение)
	Config  *Config  `json:"config,omitempty" bson:"config,omitempty"`
}

func (e Environment) Clone() *Environment {
	clone := &Environment{
		ID:          e.ID,
		SpaceID:     e.SpaceID,
		Description: e.Description,
		Aliases:     append([]string(nil), e.Aliases...),
		Config:      nil,
	}

	if e.StateInfo != nil {
		clone.StateInfo = &StateInfo{
			State:     e.StateInfo.State,
			Info:      e.StateInfo.Info,
			StartedAt: e.StateInfo.StartedAt,
		}
	}

	if e.Config != nil {
		clone.Config = &Config{
			SourceID: e.Config.SourceID,
		}
	}

	return clone
}

func (e Environment) Fetch(i interface{}) interface{} {
	p, _ := i.(string)
	switch p {
	case "ID":
		return e.ID
	case "SpaceID":
		return e.SpaceID
	case "Description":
		return e.Description
	case "StateInfo":
		return e.StateInfo
	case "Aliases":
		return e.Aliases
	case "Config":
		return e.Config
	default:
		panic("unknown parameter")
	}
}
