package data

import (
	"fmt"

	"gopkg.in/yaml.v3"
)

func Contains[T comparable](s T, arr []T) bool {

	for _, elem := range arr {
		if elem == s {
			return true
		}
	}

	return false
}

func ElementsMatch[T comparable](s1, s2 []T) bool {
	if len(s1) != len(s2) {
		return false
	}
	exists := make(map[T]bool)
	for _, value := range s1 {
		exists[value] = true
	}
	for _, value := range s2 {
		if !exists[value] {
			return false
		}
	}
	return true
}

// Difference returns the elements in `a` that aren't in `b`.
func Difference(a, b []string) []string {
	mb := make(map[string]struct{}, len(b))
	for _, x := range b {
		mb[x] = struct{}{}
	}
	var diff []string
	for _, x := range a {
		if _, found := mb[x]; !found {
			diff = append(diff, x)
		}
	}
	return diff
}

// GetIntersection returns `a` elements that are in `b`.
func GetIntersection[T comparable](a, b []T) []T {
	mb := make(map[T]struct{}, len(b))
	for _, x := range b {
		mb[x] = struct{}{}
	}
	var match []T
	for _, x := range a {
		if _, found := mb[x]; found {
			match = append(match, x)
		}
	}
	return match
}

func GetXOR[T comparable](s1, s2 []T) []T {

	res := make([]T, 0)

	m1 := make(map[T]struct{}, len(s1))
	m2 := make(map[T]struct{}, len(s2))

	for _, s := range s1 {
		m1[s] = struct{}{}
	}

	for _, s := range s2 {
		if _, ok := m1[s]; !ok {
			res = append(res, s)
		}
		m2[s] = struct{}{}
	}

	for _, s := range s1 {
		if _, ok := m2[s]; !ok {
			res = append(res, s)
		}
	}

	return res
}

func SetFromSlice[T comparable](input []T) []T {
	s := make([]T, 0, len(input))
	m := make(map[T]bool)

	for _, val := range input {
		if _, ok := m[val]; !ok {
			m[val] = true
			s = append(s, val)
		}
	}

	return s
}

func ToSliceOfInterfaces[T any](s []T) []interface{} {
	if s == nil {
		return nil
	}
	res := make([]interface{}, len(s))
	for i, e := range s {
		res[i] = e
	}
	return res
}

func ToSliceOfStrings(s []any) []string {
	if s == nil {
		return nil
	}
	res := make([]string, len(s))
	for i, v := range s {
		res[i] = fmt.Sprintf("%v", v)
	}
	return res
}

func MergeYaml(source, override string) (string, error) {
	if source == "" {
		return override, nil
	}
	if override == "" {
		return source, nil
	}

	var srcYaml map[string]interface{}
	err := yaml.Unmarshal([]byte(source), &srcYaml)
	if err != nil {
		return "", err
	}

	var overrideYaml map[string]interface{}
	err = yaml.Unmarshal([]byte(override), &overrideYaml)
	if err != nil {
		return "", err
	}

	mergedMap := mergeMaps(srcYaml, overrideYaml)

	mergedString, err := yaml.Marshal(mergedMap)
	if err != nil {
		return "", err
	}

	return string(mergedString), nil
}

func mergeMaps(a, b map[string]interface{}) map[string]interface{} {
	out := make(map[string]interface{}, len(a))
	for k, v := range a {
		out[k] = v
	}
	for k, v := range b {
		if v, ok := v.(map[string]interface{}); ok {
			if av, ok := out[k]; ok {
				if av, ok := av.(map[string]interface{}); ok {
					out[k] = mergeMaps(av, v)
					continue
				}
			}
		}

		if v, ok := v.([]interface{}); ok {
			if av, ok := out[k]; ok {
				if av, ok := av.([]interface{}); ok {
					out[k] = append(av, v...)
					continue
				}
			}
		}

		out[k] = v
	}
	return out
}
