package data

import (
	"strconv"
	"strings"
)

const DefaultFieldDelimiter = "."

type DeleteValueType struct{}

var DeleteValue DeleteValueType

// TODO: везде добавить поддержку массивов и массивов объектов

// Сделано на базе библиотеки https://github.com/knadh/koanf

// Flatten takes a map[string]interface{} and traverses it and flattens
// nested children into keys delimited by delim.
//
// It's important to note that all nested maps should be
// map[string]interface{} and not map[interface{}]interface{}.
// Use IntfaceKeysToStrings() to convert if necessary.
//
// eg: `{ "parent": { "child": 123 }}` becomes `{ "parent.child": 123 }`
// In addition, it keeps track of and returns a map of the delimited keypaths with
// a slice of key parts, for eg: { "parent.child": ["parent", "child"] }. This
// parts list is used to remember the key path's original structure to
// unflatten later.
func Flatten(m map[string]interface{}, keys []string, delim string) (map[string]interface{}, map[string][]string) {
	var (
		out    = make(map[string]interface{})
		keyMap = make(map[string][]string)
	)

	flatten(m, keys, delim, out, keyMap)
	return out, keyMap
}

func flatten(m map[string]interface{}, keys []string, delim string, out map[string]interface{}, keyMap map[string][]string) {
	for key, val := range m {
		// Copy the incoming key paths into a fresh list
		// and append the current key in the iteration.
		kp := make([]string, 0, len(keys)+1)
		kp = append(kp, keys...)
		kp = append(kp, key)

		switch cur := val.(type) {
		case map[string]interface{}:
			// Empty map.
			if len(cur) == 0 {
				newKey := strings.Join(kp, delim)
				out[newKey] = val
				keyMap[newKey] = kp
				continue
			}

			// It's a nested map. Flatten it recursively.
			flatten(cur, kp, delim, out, keyMap)
		default:
			newKey := strings.Join(kp, delim)
			out[newKey] = val
			keyMap[newKey] = kp
		}
	}
}

// Unflatten takes a flattened key:value map (non-nested with delimited keys)
// and returns a nested map where the keys are split into hierarchies by the given
// delimiter. For instance, `parent.child.key: 1` to `{parent: {child: {key: 1}}}`
//
// It's important to note that all nested maps should be
// map[string]interface{} and not map[interface{}]interface{}.
// Use IntfaceKeysToStrings() to convert if necessary.
func Unflatten(m map[string]interface{}, delim string) map[string]interface{} {
	out := make(map[string]interface{})

	// Iterate through the flat conf map.
	for k, v := range m {
		var (
			keys = strings.Split(k, delim)
			next = out
		)

		// Iterate through key parts, for eg:, parent.child.key
		// will be ["parent", "child", "key"]
		for _, k := range keys[:len(keys)-1] {
			sub, ok := next[k]
			if !ok {
				// If the key does not exist in the map, create it.
				sub = make(map[string]interface{})
				next[k] = sub
			}
			if n, ok := sub.(map[string]interface{}); ok {
				next = n
			}
		}

		// Assign the value.
		next[keys[len(keys)-1]] = v
	}
	return out
}

// Delete removes the entry present at a given path, from the interface
// if it is an object or an array.
// The path is the key map slice, for eg:, parent.child.key -> [parent child key].
// Any empty, nested map on the path, is recursively deleted.
//
// It's important to note that all nested maps should be
// map[string]interface{} and not map[interface{}]interface{}.
// Use IntfaceKeysToStrings() to convert if necessary.
func Delete(field string, data any, delim ...string) error {
	return set(getPath(field, delim...), data, DeleteValue)
}

// DeleteMany removes the entries present at a given paths, from the interface
func DeleteMany(paths []string, value any, delim ...string) {
	if value == nil || len(paths) == 0 {
		return
	}
	for _, path := range paths {
		Delete(path, value, delim...)
	}
}

// Search recursively searches the interface for a given path. The path is
// the key map slice, for eg:, parent.child.key -> [parent child key].
//
// It's important to note that all nested maps should be
// map[string]interface{} and not map[interface{}]interface{}.
// Use IntfaceKeysToStrings() to convert if necessary.
func Search(in interface{}, path []string) interface{} {
	switch val := in.(type) {

	case map[string]interface{}:
		next, ok := val[path[0]]
		if ok {
			if len(path) == 1 {
				return next
			}
			switch v := next.(type) {
			case map[string]interface{}, []interface{}:
				return Search(v, path[1:])
			}
		}
	case []interface{}:
		out := make([]interface{}, len(val))
		for i, e := range val {
			out[i] = Search(e, path)
		}
		return out
	}
	return nil
}

func getPath(field string, delim ...string) []string {
	if field == "" {
		return nil
	}

	d := DefaultFieldDelimiter
	if len(delim) > 0 {
		d = delim[0]
	}
	return strings.Split(field, d)
}

func Set(field string, data, value any, delim ...string) error {
	return set(getPath(field, delim...), data, value)
}

func set(path []string, data, value any) error {
	if len(path) == 0 {
		return nil
	}

	switch v := data.(type) {
	case map[string]interface{}:
		if len(path) == 1 {

			if _, ok := value.(DeleteValueType); ok {
				delete(v, path[0])
				return nil
			}

			v[path[0]] = value
			return nil
		}

		next, ok := v[path[0]]
		if !ok {
			next = make(map[string]interface{})
			v[path[0]] = next
		}
		return set(path[1:], next, value)

	case []interface{}:
		idx, err := strconv.Atoi(path[0])
		if err != nil {
			for _, vv := range v {
				if err = set(path, vv, value); err != nil {
					return err
				}
			}
		}
		if idx >= len(v) {
			return nil
		}
		return set(path[1:], v[idx], value)
	}

	return nil
}

func Get(field string, data any, delim ...string) (any, bool) {
	return get(getPath(field, delim...), data)
}

func get(path []string, data any) (any, bool) {
	if len(path) == 0 {
		return data, true
	}

	switch v := data.(type) {
	case map[string]interface{}:
		val, ok := v[path[0]]
		if !ok {
			return nil, false
		}
		return get(path[1:], val)
	case []interface{}:
		idx, err := strconv.Atoi(path[0])
		if err != nil || idx >= len(v) {
			return nil, false
		}
		return get(path[1:], v[idx])
	}

	return nil, false
}

// Keep keeps the entries present at a given paths, from the interface and remove other data
// if it is an object or an array.
// The path is the sting with delim, for eg:, parent.child.key
func Keep(paths []string, data any, delim ...string) {
	if len(paths) == 0 {
		data = nil
		return
	}
	switch val := data.(type) {
	case map[string]interface{}:
		for k, v := range val {
			if Contains(k, paths) {
				continue
			}
			p := getObjectPaths(k, paths, delim...)
			if len(p) == 0 {
				delete(val, k)
			}
			Keep(p, v, delim...)
		}
	case []interface{}:
		for _, ar := range val {
			Keep(paths, ar, delim...)
		}
	}
}

func getObjectPaths(prefix string, arr []string, delim ...string) []string {
	var res []string
	d := DefaultFieldDelimiter
	if len(delim) > 0 {
		d = delim[0]
	}
	for _, v := range arr {
		if strings.HasPrefix(v, prefix+d) {
			res = append(res, strings.TrimPrefix(v, prefix+d))
		}
	}
	return res
}

func CloneMap(m map[string]interface{}) map[string]interface{} {
	if m == nil {
		return m
	}

	c := make(map[string]interface{}, len(m))
	for k, v := range m {
		c[k] = v
	}
	return c
}
