package content

import (
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	clientsSvc "git.perx.ru/perxis/perxis-go/pkg/clients/middleware"
	clientsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/clients/transport/grpc"
	collaboratorsSvc "git.perx.ru/perxis/perxis-go/pkg/collaborators/middleware"
	collaboratorsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/collaborators/transport/grpc"
	collectionsSvc "git.perx.ru/perxis/perxis-go/pkg/collections/middleware"
	collectionsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/collections/transport/grpc"
	environmentsSvc "git.perx.ru/perxis/perxis-go/pkg/environments/middleware"
	environmentsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/environments/transport/grpc"
	invitationsSvc "git.perx.ru/perxis/perxis-go/pkg/invitations/middleware"
	invitationsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/invitations/transport/grpc"
	itemsSvc "git.perx.ru/perxis/perxis-go/pkg/items/middleware"
	itemsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/items/transport/grpc"
	localesSvc "git.perx.ru/perxis/perxis-go/pkg/locales/middleware"
	localsTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/locales/transport/grpc"
	referencesSvc "git.perx.ru/perxis/perxis-go/pkg/references/middleware"
	referencesTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/references/transport/grpc"
	rolesSvc "git.perx.ru/perxis/perxis-go/pkg/roles/middleware"
	rolesTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/roles/transport/grpc"
	spacesSvc "git.perx.ru/perxis/perxis-go/pkg/spaces/middleware"
	spacesTransportGrpc "git.perx.ru/perxis/perxis-go/pkg/spaces/transport/grpc"
	"go.uber.org/zap"
	"google.golang.org/grpc"
)

const (
	DefaultCacheSize = 1000
	DefaultCacheTTL  = time.Second * 10
)

func NewClient(conn *grpc.ClientConn, opts ...Option) *Content {

	client := &Content{}
	config := &Config{}

	for _, o := range opts {
		o(config)
	}

	if config.Logger == nil {
		config.Logger = zap.NewNop()
	}

	client.Spaces = spacesTransportGrpc.NewClient(conn, config.ClientOptions...)
	client.Environments = environmentsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Collections = collectionsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Items = itemsTransportGrpc.NewClient(conn, config.ClientOptions...)
	client.Invitations = invitationsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Collaborators = collaboratorsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Clients = clientsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Locales = localsTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.Roles = rolesTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)
	client.References = referencesTransportGrpc.NewGRPCClient(conn, "", config.ClientOptions...)

	if !config.NoDecode {
		client.Items = itemsSvc.ClientEncodeMiddleware(client.Collections)(client.Items)
		client.References = referencesSvc.ClientEncodeMiddleware(client.Collections)(client.References)
	}

	if !config.NoCache {
		client = WithCaching(client, DefaultCacheSize, DefaultCacheTTL)
	}

	if !config.NoLog {
		client = WithLogging(client, config.Logger, config.AccessLog)
	}

	return client
}

func WithCaching(client *Content, size int, ttl time.Duration) *Content {
	c := *client

	c.Clients = clientsSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Clients)
	c.Environments = environmentsSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Environments)
	c.Locales = localesSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Locales)
	c.Roles = rolesSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Roles)
	c.Spaces = spacesSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Spaces)
	c.Items = itemsSvc.CachingMiddleware(cache.NewCache(size, ttl), cache.NewCache(size, ttl), c.Environments)(client.Items)
	c.Collections = collectionsSvc.CachingMiddleware(cache.NewCache(size, ttl), c.Environments)(client.Collections)
	c.Collaborators = collaboratorsSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Collaborators)
	c.Invitations = invitationsSvc.CachingMiddleware(cache.NewCache(size, ttl))(client.Invitations)

	return &c
}

func WithLogging(cs *Content, logger *zap.Logger, accessLog bool) *Content {
	s := *cs

	s.Collaborators = collaboratorsSvc.WithLog(s.Collaborators, logger, accessLog)
	s.Collections = collectionsSvc.WithLog(s.Collections, logger, accessLog)
	s.Environments = environmentsSvc.WithLog(s.Environments, logger, accessLog)
	s.Invitations = invitationsSvc.WithLog(s.Invitations, logger, accessLog)
	s.Items = itemsSvc.WithLog(s.Items, logger, accessLog)
	s.Locales = localesSvc.WithLog(s.Locales, logger, accessLog)
	s.Roles = rolesSvc.WithLog(s.Roles, logger, accessLog)
	s.Spaces = spacesSvc.WithLog(s.Spaces, logger, accessLog)
	s.Clients = clientsSvc.WithLog(s.Clients, logger, accessLog)
	s.References = referencesSvc.WithLog(s.References, logger, accessLog)

	return &s
}
