package middleware

import (
	"context"
	"testing"
	"time"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	"git.perx.ru/perxis/perxis-go/pkg/collections"
	colsmocks "git.perx.ru/perxis/perxis-go/pkg/collections/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/environments"
	envmocks "git.perx.ru/perxis/perxis-go/pkg/environments/mocks"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
)

func TestCollections_Cache(t *testing.T) {

	const (
		colID    = "colID"
		spaceID  = "spaceID"
		envID    = "envId"
		envAlias = "envAlias"
		size     = 5
		ttl      = 20 * time.Millisecond
	)

	errNotFound := errors.NotFound(errors.New("not found"))

	ctx := context.Background()

	t.Run("Get from cache", func(t *testing.T) {
		col := &colsmocks.Collections{}
		env := &envmocks.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

		env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
		col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()

		v1, err := svc.Get(ctx, spaceID, envID, colID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID, envID, colID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кеша при повторном запросе по ID окружения.")

		v3, err := svc.Get(ctx, spaceID, envAlias, colID)
		require.NoError(t, err)
		assert.Same(t, v3, v2, "Ожидается получение объекта из кеша, при запросе того же объекта по alias окружения.")

		env.AssertExpectations(t)
		col.AssertExpectations(t)
	})

	t.Run("Get from cache(by Alias)", func(t *testing.T) {
		col := &colsmocks.Collections{}
		env := &envmocks.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

		col.On("Get", mock.Anything, spaceID, envAlias, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()
		env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()

		v1, err := svc.Get(ctx, spaceID, envAlias, colID)
		require.NoError(t, err)

		v2, err := svc.Get(ctx, spaceID, envAlias, colID)
		require.NoError(t, err)
		assert.Same(t, v1, v2, "Ожидается получение объекта из кеша при повторном запросе по Alias окружения.")

		v3, err := svc.Get(ctx, spaceID, envID, colID)
		require.NoError(t, err)
		assert.Same(t, v3, v2, "Ожидается получение объекта из кеша, при запросе того же объекта по ID окружения.")

		env.AssertExpectations(t)
		col.AssertExpectations(t)
	})

	t.Run("Get from cache with options", func(t *testing.T) {
		col := &colsmocks.Collections{}
		env := &envmocks.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

		env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
		col.On("Get", mock.Anything, spaceID, envID, colID, mock.Anything).Run(func(args mock.Arguments) {
			require.Len(t, args, 5)
			opt := args.Get(4).(*collections.GetOptions)
			assert.True(t, opt.DisableSchemaIncludes)
		}).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()

		_, err := svc.Get(ctx, spaceID, envID, colID, []*collections.GetOptions{{DisableSchemaIncludes: true}}...)
		require.NoError(t, err)

		env.AssertExpectations(t)
		col.AssertExpectations(t)
	})

	//t.Run("List from cache", func(t *testing.T) {
	//	col := &colsmocks.Collections{}
	//	env := &envmocks.Environments{}
	//
	//	svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)
	//
	//	col.On("List", mock.Anything, spaceID, envID).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()
	//	env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
	//
	//	vl1, err := svc.List(ctx, spaceID, envID, nil)
	//	require.NoError(t, err)
	//
	//	vl2, err := svc.List(ctx, spaceID, envID, nil)
	//	require.NoError(t, err)
	//	assert.Len(t, vl2, 1)
	//	assert.Same(t, vl1[0], vl2[0], "При повторном запросе по ID окружения, ожидается получение списка объектов из кеша.")
	//
	//	vl3, err := svc.List(ctx, spaceID, envAlias, nil)
	//	require.NoError(t, err)
	//	assert.Len(t, vl3, 1)
	//	assert.Same(t, vl3[0], vl2[0], "При повторном запросе по Alias окружения, ожидается получение списка объектов из кеша.")
	//
	//	env.AssertExpectations(t)
	//	col.AssertExpectations(t)
	//})

	t.Run("List", func(t *testing.T) {
		col := &colsmocks.Collections{}
		env := &envmocks.Environments{}

		svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

		col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()
		col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

		_, err := svc.List(ctx, spaceID, envAlias, nil)
		require.NoError(t, err)

		_, err = svc.List(ctx, spaceID, envID, nil)
		require.NoError(t, err)

		env.AssertExpectations(t)
		col.AssertExpectations(t)
	})

	t.Run("Invalidate cache", func(t *testing.T) {
		t.Run("After Update", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кеша по ID окружения.")

			v3, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кеша по Alias окружения.")

			vl1, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Update", mock.Anything, mock.Anything).Return(nil).Once()
			err = svc.Update(ctx, &collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"})
			require.NoError(t, err)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}}, nil).Once()

			v4, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.NotSame(t, v3, v4, "Ожидает что элемент после обновления был удален из кэша и будет запрошен заново из сервиса.")

			v5, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v4, v5, "Ожидается получение объекта из кеша по Alias окружения.")

			vl2, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.NotSame(t, vl1[0], vl2[0], "Ожидает что после обновления элементы будут запрошены заново из сервиса.")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})

		t.Run("After Update(by Alias)", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			//env.On("Get", mock.Anything, spaceID, envAlias).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()

			col.On("Get", mock.Anything, spaceID, envAlias, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кеша по Alias окружения.")

			v3, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кеша по ID окружения.")

			vl1, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Update", mock.Anything, mock.Anything).Return(nil).Once()
			err = svc.Update(ctx, &collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"})
			require.NoError(t, err)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			//env.On("Get", mock.Anything, spaceID, envAlias).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()

			col.On("Get", mock.Anything, spaceID, envAlias, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}}, nil).Once()

			v4, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.NotSame(t, v3, v4, "Ожидает что элемент после обновления был удален из кэша и будет запрошен заново из сервиса.")

			v5, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v4, v5, "Ожидается получение объекта из кеша по Alias окружения.")

			vl4, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.NotSame(t, vl1[0], vl4[0], "Ожидает что после обновления элементы будут запрошены заново из сервиса.")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})

		t.Run("After Set Schema", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Twice()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Twice()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кеша по ID окружения.")

			v3, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кеша по Alias окружения.")

			vl1, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кеша по ID окружения.")

			vl3, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Equal(t, vl2[0], vl3[0], "Ожидается получение объектов из кеша по Alias окружения.")

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("SetSchema", mock.Anything, spaceID, envID, colID, mock.Anything).Return(nil).Once()
			err = svc.SetSchema(ctx, spaceID, envID, colID, &schema.Schema{})
			require.NoError(t, err)

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}}, nil).Once()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "nameUPD"}}, nil).Once()

			v4, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.NotSame(t, v3, v4, "Ожидает что элемент после обновления схемы был удален из кэша и будет запрошен заново из сервиса.")

			v5, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v4, v5, "Ожидается получение объекта из кеша по Alias окружения.")

			vl4, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.NotSame(t, vl4[0], vl3[0], "Ожидает что после обновления схемы элементы будут запрошены заново из сервиса.")

			vl5, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.Equal(t, vl4[0], vl5[0], "Ожидается получение объектов из кеша по Alias окружения..")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})

		t.Run("After Delete", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Twice()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Twice()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кеша по ID окружения.")

			v3, err := svc.Get(ctx, spaceID, envAlias, colID)
			require.NoError(t, err)
			assert.Same(t, v2, v3, "Ожидается получение объекта из кеша по Alias окружения.")

			vl1, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кеша по ID окружения.")

			vl3, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Equal(t, vl2[0], vl3[0], "Ожидается получение объектов из кеша по Alias окружения.")

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Delete", mock.Anything, spaceID, envID, colID).Return(nil).Once()
			err = svc.Delete(ctx, spaceID, envID, colID)
			require.NoError(t, err)

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(nil, errNotFound).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{}, nil).Once()

			_, err = svc.Get(ctx, spaceID, envID, colID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидает что элемент был удален из кэша и получена ошибка от сервиса.")

			vl4, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.Len(t, vl4, 0, "Ожидает что элементы были удалены из кэша.")

			col.On("Get", mock.Anything, spaceID, envAlias, colID).Return(nil, errNotFound).Once()

			_, err = svc.Get(ctx, spaceID, envAlias, colID)
			require.Error(t, err)
			assert.EqualError(t, err, "not found", "Ожидает что элемент был удален из кэша и получена ошибка от сервиса.")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})

		t.Run("After Create", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Twice()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}}, nil).Once()

			vl1, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)

			vl2, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Same(t, vl1[0], vl2[0], "Ожидается получение объектов из кеша по ID окружения.")

			vl3, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.Len(t, vl2, 1)
			assert.Equal(t, vl2[0], vl3[0], "Ожидается получение объектов из кеша по Alias окружения.")

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("Create", mock.Anything, mock.Anything).Return(&collections.Collection{ID: "colID2", SpaceID: spaceID, EnvID: envID, Name: "name2"}, nil).Once()
			_, err = svc.Create(ctx, &collections.Collection{ID: "colID2", SpaceID: spaceID, EnvID: envID, Name: "name2"})
			require.NoError(t, err)

			//env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil).Once()
			col.On("List", mock.Anything, spaceID, envID, mock.Anything).Return([]*collections.Collection{
				{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"},
				{ID: "colID2", SpaceID: spaceID, EnvID: envID, Name: "name2"},
			}, nil).Once()
			col.On("List", mock.Anything, spaceID, envAlias, mock.Anything).Return([]*collections.Collection{
				{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"},
				{ID: "colID2", SpaceID: spaceID, EnvID: envID, Name: "name2"},
			}, nil).Once()

			vl4, err := svc.List(ctx, spaceID, envID, nil)
			require.NoError(t, err)
			assert.Len(t, vl4, 2, "Ожидает что элементы были удалены из кэша и получены заново из сервиса.")

			vl5, err := svc.List(ctx, spaceID, envAlias, nil)
			require.NoError(t, err)
			assert.Len(t, vl5, 2)
			assert.Equal(t, vl4[0], vl5[0], "Ожидается получение объектов из кеша по Alias окружения..")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})

		t.Run("After TTL expired", func(t *testing.T) {
			col := &colsmocks.Collections{}
			env := &envmocks.Environments{}

			svc := CachingMiddleware(cache.NewCache(size, ttl), env)(col)

			env.On("Get", mock.Anything, spaceID, envID).Return(&environments.Environment{ID: envID, SpaceID: spaceID, Aliases: []string{envAlias}}, nil)
			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()

			v1, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)

			v2, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.Same(t, v1, v2, "Ожидается получение объекта из кеша.")

			time.Sleep(2 * ttl)

			col.On("Get", mock.Anything, spaceID, envID, colID).Return(&collections.Collection{ID: colID, SpaceID: spaceID, EnvID: envID, Name: "name"}, nil).Once()

			v3, err := svc.Get(ctx, spaceID, envID, colID)
			require.NoError(t, err)
			assert.NotSame(t, v3, v2, "Ожидает что элемент был удален из кэша и будет запрошен заново из сервиса.")

			env.AssertExpectations(t)
			col.AssertExpectations(t)
		})
	})

}
