package middleware

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/collections"
	envService "git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/schema"
)

func makeKey(spaceId, envId, collectionId string, disableSchemaIncludes bool) string {
	s := spaceId + "-" + envId + "-" + collectionId + "-"
	if disableSchemaIncludes {
		s += "1"
	} else {
		s += "0"
	}
	return s
}

func CachingMiddleware(cache *cache.Cache, envs envService.Environments) Middleware {
	return func(next service.Collections) service.Collections {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
			envs:  envs,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Collections
	envs  envService.Environments
}

func (m cachingMiddleware) Create(ctx context.Context, collection *service.Collection) (coll *service.Collection, err error) {
	return m.next.Create(ctx, collection)
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId string, envId string, collectionId string, options ...*service.GetOptions) (coll *service.Collection, err error) {

	opts := service.MergeGetOptions(options...)
	value, e := m.cache.Get(makeKey(spaceId, envId, collectionId, opts.DisableSchemaIncludes))
	if e == nil {
		return value.(*service.Collection), err
	}
	coll, err = m.next.Get(ctx, spaceId, envId, collectionId, options...)
	if err == nil {
		env, err := m.envs.Get(ctx, coll.SpaceID, coll.EnvID)
		if err != nil {
			return nil, err
		}
		m.cache.Set(makeKey(coll.SpaceID, env.ID, coll.ID, opts.DisableSchemaIncludes), coll)
		for _, al := range env.Aliases {
			m.cache.Set(makeKey(coll.SpaceID, al, coll.ID, opts.DisableSchemaIncludes), coll)
		}

	}
	return coll, err
}

func (m cachingMiddleware) List(ctx context.Context, spaceId, envId string, filter *service.Filter) (collections []*service.Collection, err error) {
	return m.next.List(ctx, spaceId, envId, filter)
}

func (m cachingMiddleware) Update(ctx context.Context, coll *service.Collection) (err error) {

	err = m.next.Update(ctx, coll)
	if err == nil {
		env, err := m.envs.Get(ctx, coll.SpaceID, coll.EnvID)
		if err != nil {
			return err
		}
		m.cache.Remove(makeKey(env.SpaceID, env.ID, coll.ID, true))
		m.cache.Remove(makeKey(env.SpaceID, env.ID, coll.ID, false))
		for _, al := range env.Aliases {
			m.cache.Remove(makeKey(env.SpaceID, al, coll.ID, true))
			m.cache.Remove(makeKey(env.SpaceID, al, coll.ID, false))
		}
	}
	return err
}

func (m cachingMiddleware) SetSchema(ctx context.Context, spaceId, envId, collectionId string, schema *schema.Schema) (err error) {
	err = m.next.SetSchema(ctx, spaceId, envId, collectionId, schema)
	if err == nil {
		env, err := m.envs.Get(ctx, spaceId, envId)
		if err != nil {
			return err
		}
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, true))
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, false))
		for _, al := range env.Aliases {
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, true))
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, false))
		}
	}
	return err
}

func (m cachingMiddleware) SetState(ctx context.Context, spaceId, envId, collectionId string, state *service.StateInfo) (err error) {
	err = m.next.SetState(ctx, spaceId, envId, collectionId, state)
	if err == nil {
		env, err := m.envs.Get(ctx, spaceId, envId)
		if err != nil {
			return err
		}
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, true))
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, false))
		for _, al := range env.Aliases {
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, true))
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, false))
		}
	}
	return err
}

func (m cachingMiddleware) Delete(ctx context.Context, spaceId string, envId string, collectionId string) (err error) {

	err = m.next.Delete(ctx, spaceId, envId, collectionId)
	if err == nil {
		env, err := m.envs.Get(ctx, spaceId, envId)
		if err != nil {
			return err
		}
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, true))
		m.cache.Remove(makeKey(env.SpaceID, env.ID, collectionId, false))
		for _, al := range env.Aliases {
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, true))
			m.cache.Remove(makeKey(env.SpaceID, al, collectionId, false))
		}
	}
	return err
}
